/**
 * Front-end JavaScript for the Volunteer Application Form plugin.
 * Handles client-side validation, date auto-population, and AJAX form submission.
 */

jQuery(document).ready(function($) {
	const $form = $('.bootstrap-volunteer-form__form'); // Target the form using its class
	const $messagesDiv = $form.find('.va-form-messages'); // Div for displaying AJAX messages
	const $submitButton = $form.find('button[type="submit"]');
	let originalButtonText = $submitButton.html(); // Store original button text

	/**
	 * Set Application Submission Date to Current Date and make it readonly.
	 */
	const $submissionDateInput = $('input[name="applicationsubmissiondate"]');
	if($submissionDateInput.length) {
		const today = new Date();
		const yyyy = today.getFullYear();
		const mm = String(today.getMonth() + 1).padStart(2, '0'); // Months are 0-indexed
		const dd = String(today.getDate()).padStart(2, '0');
		const formattedDate = `${yyyy}-${mm}-${dd}`;
		$submissionDateInput.val(formattedDate);
		$submissionDateInput.prop('readonly', true); // Ensure it's readonly
	}

	/**
	 * Custom Client-Side Validation Logic
	 */
	$form.on('submit', function(e) {
		// Prevent default HTML5 validation for custom handling
		e.preventDefault();
		e.stopPropagation();

		// Remove any existing messages and validation classes
		$messagesDiv.removeClass('alert alert-success alert-warning alert-danger').html('');
		$form.find('.is-invalid').removeClass('is-invalid');
		$form.find('.is-valid').removeClass('is-valid');

		// Custom validation for 'Days Available' checkboxes
		const $daysCheckboxes = $form.find('.days-available-group input[type="checkbox"]');
		const $daysFeedback = $form.find('.days-available-feedback');
		if($daysCheckboxes.filter(':checked').length === 0) {
			$daysCheckboxes.addClass('is-invalid');
			$daysFeedback.show();
		} else {
			$daysCheckboxes.removeClass('is-invalid').addClass('is-valid');
			$daysFeedback.hide();
		}

		// Custom validation for 'Interests and Skills' checkboxes and 'Other' textarea
		const $interestsCheckboxes = $form.find('.interests-and-skills input[type="checkbox"]');
		const $otherInterestsTextarea = $form.find('textarea[name="otherinterests"]');
		const $interestsSkillsFeedback = $form.find('.interests-skills-feedback');

		if($interestsCheckboxes.filter(':checked').length > 0 || $otherInterestsTextarea.val().trim() !== '') {
			$interestsCheckboxes.removeClass('is-invalid');
			$otherInterestsTextarea.removeClass('is-invalid');
			$interestsSkillsFeedback.hide();
		} else {
			$interestsCheckboxes.addClass('is-invalid');
			$otherInterestsTextarea.addClass('is-invalid');
			$interestsSkillsFeedback.show();
		}

		// For Certify checkbox
		const $certifyCheckbox = $form.find('#certifythattheinformationprovidedistrue');
		const $certifyFeedback = $form.find('.certify-feedback');
		if(!$certifyCheckbox.is(':checked')) {
			$certifyCheckbox.addClass('is-invalid').removeClass('is-valid');
			$certifyFeedback.show();
		} else {
			$certifyCheckbox.removeClass('is-invalid').addClass('is-valid');
			$certifyFeedback.hide();
		}

		// Apply Bootstrap's built-in validation for other required fields
		let isValid = true;
		$form.find('[required]').each(function() {
			if($(this).is(':checkbox')) {
				// Checkboxes are handled by custom logic above, skip here
				return true;
			}
			if(!this.checkValidity()) {
				$(this).addClass('is-invalid').removeClass('is-valid');
				isValid = false;
			} else {
				$(this).removeClass('is-invalid').addClass('is-valid');
			}
		});

		// Re-check overall form validity after custom checks
		if($daysCheckboxes.filter(':checked').length === 0 ||
			($interestsCheckboxes.filter(':checked').length === 0 && $otherInterestsTextarea.val().trim() === '') ||
			!$certifyCheckbox.is(':checked')) {
			isValid = false;
		}

		if(isValid) {
			// If all client-side validations pass, proceed with AJAX submission
			$form.addClass('was-validated'); // Ensure validation styles are shown
			$submitButton.prop('disabled', true).html(va_ajax_object.messages.submitting); // Disable button and show submitting text

			const formData = $form.serializeArray();
			// Add nonce to form data
			formData.push({name: 'action', value: 'va_submit_volunteer_form'});
			formData.push({name: 'volunteer_form_nonce_field', value: va_ajax_object.nonce});

			// Perform AJAX request
			$.ajax({
				url: va_ajax_object.ajax_url, // WordPress AJAX URL
				type: 'POST',
				data: formData,
				dataType: 'json', // Expect JSON response from the server
				success: function(response) {
					// Always remove previous classes before adding new ones
					$messagesDiv.removeClass('alert alert-success alert-warning alert-danger');

					if(response.success) {
						// If the server-side processing was successful (wp_send_json_success)
						$messagesDiv.addClass('alert alert-success');
						$messagesDiv.html(response.data.message);
						$form[0].reset(); // Clear the form fields
						// Reset custom validation states
						$form.find('.is-valid').removeClass('is-valid');
						$form.find('.is-invalid').removeClass('is-invalid');
						$daysFeedback.hide();
						$interestsSkillsFeedback.hide();
						$certifyFeedback.hide();
						$form.removeClass('was-validated'); // Remove validation feedback
					} else {
						// If the server-side processing returned an error (wp_send_json_error)
						const alertType = response.data.type || 'danger'; // Default to 'danger' if type is not specified.
						$messagesDiv.addClass('alert alert-' + alertType);
						$messagesDiv.html(response.data.message);
						$form.addClass('was-validated'); // Keep validation feedback visible on error.
					}
				},
				error: function(jqXHR, textStatus, errorThrown) {
					console.error('AJAX Error:', textStatus, errorThrown, jqXHR);
					$messagesDiv.addClass('alert alert-danger').html(va_ajax_object.messages.error);
					$form.addClass('was-validated'); // Keep validation feedback visible on error.
				},
				complete: function() {
					// Re-enable the submit button and restore its original text.
					$submitButton.prop('disabled', false).html(originalButtonText);
				}
			});
		} else {
			// If client-side validation fails, add Bootstrap's 'was-validated' class to show feedback
			$form.addClass('was-validated');
			// Display a generic validation error message if not already specific
			if($messagesDiv.html() === '') {
				$messagesDiv.removeClass('alert alert-success alert-warning alert-danger').addClass('alert alert-danger').html(va_ajax_object.messages.validation_error);
			}
		}
	});

	// Handle Bootstrap's default validation classes for other inputs on change/blur
	$form.find('input, select, textarea').not('.form-check-input').on('change blur', function() {
		if(this.checkValidity()) {
			$(this).removeClass('is-invalid').addClass('is-valid');
		} else {
			$(this).addClass('is-invalid').removeClass('is-valid');
		}
	});

	// Remove validation feedback when input is corrected for checkboxes
	$form.find('.days-available-group input[type="checkbox"]').on('change', function() {
		const $daysCheckboxes = $form.find('.days-available-group input[type="checkbox"]');
		const $daysFeedback = $form.find('.days-available-feedback');
		if($daysCheckboxes.filter(':checked').length > 0) {
			$daysCheckboxes.removeClass('is-invalid').addClass('is-valid');
			$daysFeedback.hide();
		} else {
			$daysCheckboxes.addClass('is-invalid').removeClass('is-valid');
			$daysFeedback.show();
		}
	});

	$form.find('.interests-and-skills input[type="checkbox"], textarea[name="otherinterests"]').on('change blur', function() {
		const $interestsCheckboxes = $form.find('.interests-and-skills input[type="checkbox"]');
		const $otherInterestsTextarea = $form.find('textarea[name="otherinterests"]');
		const $interestsSkillsFeedback = $form.find('.interests-skills-feedback');

		if($interestsCheckboxes.filter(':checked').length > 0 || $otherInterestsTextarea.val().trim() !== '') {
			$interestsCheckboxes.removeClass('is-invalid');
			$otherInterestsTextarea.removeClass('is-invalid');
			$interestsSkillsFeedback.hide();
		} else {
			$interestsCheckboxes.addClass('is-invalid');
			$otherInterestsTextarea.addClass('is-invalid');
			$interestsSkillsFeedback.show();
		}
	});

	// For Certify checkbox
	$form.find('#certifythattheinformationprovidedistrue').on('change', function() {
		const $certifyCheckbox = $(this);
		const $certifyFeedback = $form.find('.certify-feedback');
		if($certifyCheckbox.is(':checked')) {
			$certifyCheckbox.removeClass('is-invalid').addClass('is-valid');
			$certifyFeedback.hide();
		} else {
			$certifyCheckbox.addClass('is-invalid').removeClass('is-valid');
			$certifyFeedback.show();
		}
	});

	// Handle Bootstrap's default validation classes for other inputs on change/blur
	$form.find('input, select, textarea').not('.form-check-input').on('change blur', function() {
		if(this.checkValidity()) {
			$(this).removeClass('is-invalid').addClass('is-valid');
		} else {
			$(this).addClass('is-invalid').removeClass('is-valid');
		}
	});
});