<?php
/**
 * Plugin Name: Bootstrap Partner Inquiry
 * Plugin URI:  https://www.emad-zedan.com
 * Description: Provides a comprehensive partner inquiry form with a custom post type to store submissions, client-side jQuery validation, AJAX submission, and dynamic email notifications. Requires Advanced Custom Fields (ACF) plugin (free version compatible) for data storage.
 * Version:     1.0.0
 * Requires at least: 5.2
 * Requires PHP: 8.0
 * Author:      Emad Zedan
 * Author URI:  https://www.emad-zedan.com
 * License:     GPL2
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: bootstrap-partner-inquiry
 * Domain Path: /languages
 */

// Exit if accessed directly to prevent unauthorized access.
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

// Define plugin constants for easier path management.
define( 'BPI_PLUGIN_DIR', plugin_dir_path( __FILE__ ) );
define( 'BPI_PLUGIN_URL', plugin_dir_url( __FILE__ ) );

// Admin CPT Registration
add_action('init', 'bpi_create_partner_cpt');

// Admin CSS & JS Enqueue
add_action('admin_enqueue_scripts', 'bpi_admin_enqueue_assets');

// Front-end CSS & JS Enqueue
add_action('wp_enqueue_scripts', 'bpi_frontend_enqueue_assets');

// Front-end Shortcode
add_shortcode('partner_inquiry_form', 'bpi_render_partner_form_shortcode');

// AJAX Handler for form submission
add_action('wp_ajax_bpi_submit_form', 'bpi_handle_ajax_form_submission');
add_action('wp_ajax_nopriv_bpi_submit_form', 'bpi_handle_ajax_form_submission');

// Admin CPT: Add custom columns to list table
add_filter('manage_partner_posts_columns', 'bpi_add_partner_columns');
add_action('manage_partner_posts_custom_column', 'bpi_populate_partner_columns', 10, 2);

// Add new submenu pages under the 'Partner Inquiries' CPT.
add_action('admin_menu', 'bpi_add_settings_pages');
add_action('admin_init', 'bpi_register_settings');

// Add dashboard widget
add_action('wp_dashboard_setup', 'bpi_add_dashboard_widgets');


/**
 * Register Custom Post Type: Partners (partner)
 * This CPT will store all partner inquiry submissions.
 */
function bpi_create_partner_cpt(){
    $labels = array(
        'name'                  => _x( 'Partner Inquiries', 'Post Type General Name', 'bootstrap-partner-inquiry' ),
        'singular_name'         => _x( 'Partner Inquiry', 'Post Type Singular Name', 'bootstrap-partner-inquiry' ),
        'menu_name'             => __( 'Partner Inquiries', 'bootstrap-partner-inquiry' ),
        'name_admin_bar'        => __( 'Partner Inquiry', 'bootstrap-partner-inquiry' ),
        'archives'              => __( 'Inquiry Archives', 'bootstrap-partner-inquiry' ),
        'attributes'            => __( 'Inquiry Attributes', 'bootstrap-partner-inquiry' ),
        'parent_item_colon'     => __( 'Parent Inquiry:', 'bootstrap-partner-inquiry' ),
        'all_items'             => __( 'All Inquiries', 'bootstrap-partner-inquiry' ),
        'add_new_item'          => __( 'Add New Inquiry', 'bootstrap-partner-inquiry' ),
        'add_new'               => __( 'Add New', 'bootstrap-partner-inquiry' ),
        'new_item'              => __( 'New Inquiry', 'bootstrap-partner-inquiry' ),
        'edit_item'             => __( 'Edit Inquiry', 'bootstrap-partner-inquiry' ),
        'update_item'           => __( 'Update Inquiry', 'bootstrap-partner-inquiry' ),
        'view_item'             => __( 'View Inquiry', 'bootstrap-partner-inquiry' ),
        'view_items'            => __( 'View Inquiries', 'bootstrap-partner-inquiry' ),
        'search_items'          => __( 'Search Inquiries', 'bootstrap-partner-inquiry' ),
        'not_found'             => __( 'Not found', 'bootstrap-partner-inquiry' ),
        'not_found_in_trash'    => __( 'Not found in Trash', 'bootstrap-partner-inquiry' ),
        'featured_image'        => __( 'Featured Image', 'bootstrap-partner-inquiry' ),
        'set_featured_image'    => __( 'Set featured image', 'bootstrap-partner-inquiry' ),
        'remove_featured_image' => __( 'Remove featured image', 'bootstrap-partner-inquiry' ),
        'use_featured_image'    => __( 'Use as featured image', 'bootstrap-partner-inquiry' ),
        'insert_into_item'      => __( 'Insert into inquiry', 'bootstrap-partner-inquiry' ),
        'uploaded_to_this_item' => __( 'Uploaded to this inquiry', 'bootstrap-partner-inquiry' ),
        'items_list'            => __( 'Inquiries list', 'bootstrap-partner-inquiry' ),
        'items_list_navigation' => __( 'Inquiries list navigation', 'bootstrap-partner-inquiry' ),
        'filter_items_list'     => __( 'Filter inquiries list', 'bootstrap-partner-inquiry' ),
    );

    $args = array(
        'label'                 => __( 'Partner Inquiry', 'bootstrap-partner-inquiry' ),
        'description'           => __( 'Partner inquiry submissions from the website.', 'bootstrap-partner-inquiry' ),
        'labels'                => $labels,
        'supports'              => array( 'title' ), // Using 'title' for organization name
        'hierarchical'          => false,
        'public'                => false, // Not publicly viewable on front-end
        'show_ui'               => true,
        'show_in_menu'          => true,
        'menu_position'         => 30, // Position in admin menu
        'menu_icon'             => 'dashicons-groups', // Icon for the menu item
        'show_in_admin_bar'     => true,
        'show_in_nav_menus'     => false, // Not for navigation menus
        'can_export'            => true,
        'has_archive'           => false,
        'exclude_from_search'   => true,
        'publicly_queryable'    => false,
        'capability_type'       => 'post',
        'rewrite'               => false,
    );
    register_post_type('partner', $args);
}

/**
 * Enqueue Admin CSS & JS
 * Enqueues styles and scripts for the WordPress admin area.
 */
function bpi_admin_enqueue_assets(){
    // No custom admin CSS/JS for this plugin yet, but hooks are ready.
}

/**
 * Enqueue Front-end CSS & JS
 * Enqueues styles and scripts for the front-end partner inquiry form.
 */
function bpi_frontend_enqueue_assets(){
    // Enqueue Bootstrap CSS from CDN
    //wp_enqueue_style('bpi-bootstrap-css', "https://cdn.jsdelivr.net/npm/bootstrap@5.3.7/dist/css/bootstrap.min.css", array(), '5.3.7', 'all');
    // Enqueue Google Fonts (Montserrat)
    //wp_enqueue_style('bpi-google-fonts', "https://fonts.googleapis.com/css2?family=Montserrat:ital,wght@0,100..900;1,100..900&display=swap", array(), null);
    // Enqueue custom CSS for the form (assuming it's in a 'style' directory within 'assets/wp/')
    wp_enqueue_style('bpi-custom-css', BPI_PLUGIN_URL . 'assets/wp/style/css/partner.min.css', array(), '1.0.5', 'all');

    // jQuery
    //wp_enqueue_script('bpi-jQuery', "https://code.jquery.com/jquery-3.7.1.min.js", array(), '3.7.1', true);
    // Enqueue Bootstrap JS bundle (contains Popper.js)
    //wp_enqueue_script('bpi-bootstrap-js', "https://cdn.jsdelivr.net/npm/bootstrap@5.3.7/dist/js/bootstrap.bundle.min.js", array(), '5.3.7', true);
    // Enqueue custom JavaScript for form validation and AJAX submission
    wp_enqueue_script('bpi-form-script', BPI_PLUGIN_URL . 'assets/wp/js/wp-js.js', array(), '1.0.0', true);

    // Localize script to pass AJAX URL and nonce to JavaScript
    wp_localize_script(
        'bpi-form-script', // Handle of the script to localize
        'bpi_ajax_object', // Name of the JS object in global scope
        array(
            'ajax_url' => admin_url( 'admin-ajax.php' ),
            'nonce'    => wp_create_nonce( 'partner_form_nonce' ), // Unique nonce for this form
            'messages' => array(
                'success'           => __( 'Thank you for your inquiry! We will get back to you shortly.', 'bootstrap-partner-inquiry' ),
                'email_error'       => __( 'Your inquiry was saved, but there was an issue sending the confirmation email. Please contact support.', 'bootstrap-partner-inquiry' ),
                'error'             => __( 'There was an error submitting your inquiry. Please ensure all fields are correctly filled and try again.', 'bootstrap-partner-inquiry' ),
                'validation_error'  => __( 'Please fill in all required fields correctly.', 'bootstrap-partner-inquiry' ),
                'security_error'    => __( 'Security check failed. Please try again.', 'bootstrap-partner-inquiry' ),
                'submitting'        => __( 'Submitting...', 'bootstrap-partner-inquiry' ),
                'org_type_required' => __( 'Please select at least one type of organization.', 'bootstrap-partner-inquiry' ),
                'collab_area_required'=> __( 'Please select at least one area of collaboration or describe in "Other".', 'bootstrap-partner-inquiry' ),
                'specific_project_desc_required' => __( 'Please describe your specific project, program, or idea.', 'bootstrap-partner-inquiry' ),
            ),
        )
    );
}

/**
 * Shortcode to display the Partner Inquiry Form.
 * Usage: [partner_inquiry_form]
 */
function bpi_render_partner_form_shortcode(){
    ob_start(); // Start output buffering to capture HTML
    ?>
    <div class="partner-form container pt-5 pb-5 px-0">
        <form method="post" class="bootstrap-partner-form__form needs-validation" novalidate>
            <h2 class="text-start"><?php _e('Partnership Inquiry Form', 'bootstrap-partner-inquiry'); ?></h2>
            <p><?php _e('Thank you for your interest in partnering with the Shako Mako Iraqi American Association. Please complete the form below, and a member of our team will follow up with you shortly.', 'bootstrap-partner-inquiry'); ?></p>
            <p class="primary-text mt-4">1. <?php _e('Organization Information', 'bootstrap-partner-inquiry'); ?></p>
            <div class="row mt-2">
                <div class="col-12 col-sm-12 col-md-12 col-lg-12 col-xl-12 col-xxl-12 mt-2">
                    <input class="form-control" type="text" name="organizationname" placeholder="<?php _e('Organization Name', 'bootstrap-partner-inquiry'); ?>" required>
                    <div class="invalid-feedback"><?php _e('Please enter your organization name', 'bootstrap-partner-inquiry'); ?></div>
                </div>
            </div>

            <p class="mb-0 mt-2"><?php _e('Type of Organization:', 'bootstrap-partner-inquiry'); ?></p>
            <div class="row">
                <div class="col-12 col-sm-12 col-md-12 col-lg-12 col-xl-12 col-xxl-12 mt-2">
                    <div class="form-check type-of-organization-group">
                        <div>
                            <input class="form-check-input" type="checkbox" name="type_of_organization[]" value="Nonprofit" id="nonprofit">
                            <label class="form-check-label" for="nonprofit"><?php _e('Nonprofit', 'bootstrap-partner-inquiry'); ?></label>
                        </div>
                        <div>
                            <input class="form-check-input" type="checkbox" name="type_of_organization[]" value="Business" id="business">
                            <label class="form-check-label" for="business"><?php _e('Business', 'bootstrap-partner-inquiry'); ?></label>
                        </div>
                        <div>
                            <input class="form-check-input" type="checkbox" name="type_of_organization[]" value="Educational Institution" id="educationalinstitution">
                            <label class="form-check-label" for="educationalinstitution"><?php _e('Educational Institution', 'bootstrap-partner-inquiry'); ?></label>
                        </div>
                        <div>
                            <input class="form-check-input" type="checkbox" name="type_of_organization[]" value="Government Agency" id="governmentagency">
                            <label class="form-check-label" for="governmentagency"><?php _e('Government Agency', 'bootstrap-partner-inquiry'); ?></label>
                        </div>

                        <div>
                            <input class="form-check-input" type="checkbox" name="type_of_organization[]" value="Media/Arts" id="mediaarts">
                            <label class="form-check-label" for="mediaarts"><?php _e('Media/Arts', 'bootstrap-partner-inquiry'); ?></label>
                        </div>
                        <div>
                            <input class="form-check-input" type="checkbox" name="type_of_organization[]" value="Other" id="other_org_type">
                            <label class="form-check-label" for="other_org_type"><?php _e('Other', 'bootstrap-partner-inquiry'); ?></label>
                        </div>
                    </div>
                    <div class="invalid-feedback organization-type-feedback"><?php _e('Please select at least one organization type', 'bootstrap-partner-inquiry'); ?></div>
                </div>
            </div>

            <p class="mb-0 mt-2"><?php _e('Website or Social Media', 'bootstrap-partner-inquiry'); ?></p>
            <div>
                <textarea class="form-control" name="websiteorsocialmedia" placeholder="<?php _e('Website or Social Media URL(s)', 'bootstrap-partner-inquiry'); ?>"></textarea>
            </div>

            <p class="mb-0 mt-2"><?php _e('Address:', 'bootstrap-partner-inquiry'); ?></p>
            <div class="row">
                <div class="col-12 col-sm-12 col-md-6 col-lg-6 col-xl-6 col-xxl-6 mt-2">
                    <select class="form-select" name="country" required>
                        <option selected disabled value><?php _e('Country', 'bootstrap-partner-inquiry'); ?></option>
                        <option value="Afganistan">Afghanistan</option>
                        <option value="Albania">Albania</option>
                        <option value="Algeria">Algeria</option>
                        <option value="American Samoa">American Samoa</option>
                        <option value="Andorra">Andorra</option>
                        <option value="Angola">Angola</option>
                        <option value="Anguilla">Anguilla</option>
                        <option value="Antigua & Barbuda">Antigua & Barbuda</option>
                        <option value="Argentina">Argentina</option>
                        <option value="Armenia">Armenia</option>
                        <option value="Aruba">Aruba</option>
                        <option value="Australia">Australia</option>
                        <option value="Austria">Austria</option>
                        <option value="Azerbaijan">Azerbaijan</option>
                        <option value="Bahamas">Bahamas</option>
                        <option value="Bahrain">Bahrain</option>
                        <option value="Bangladesh">Bangladesh</option>
                        <option value="Barbados">Barbados</option>
                        <option value="Belarus">Belarus</option>
                        <option value="Belgium">Belgium</option>
                        <option value="Belize">Belize</option>
                        <option value="Benin">Benin</option>
                        <option value="Bermuda">Bermuda</option>
                        <option value="Bhutan">Bhutan</option>
                        <option value="Bolivia">Bolivia</option>
                        <option value="Bonaire">Bonaire</option>
                        <option value="Bosnia & Herzegovina">Bosnia & Herzegovina</option>
                        <option value="Botswana">Botswana</option>
                        <option value="Brazil">Brazil</option>
                        <option value="British Indian Ocean Ter">British Indian Ocean Ter</option>
                        <option value="Brunei">Brunei</option>
                        <option value="Bulgaria">Bulgaria</option>
                        <option value="Burkina Faso">Burkina Faso</option>
                        <option value="Burundi">Burundi</option>
                        <option value="Cambodia">Cambodia</option>
                        <option value="Cameroon">Cameroon</option>
                        <option value="Canada">Canada</option>
                        <option value="Canary Islands">Canary Islands</option>
                        <option value="Cape Verde">Cape Verde</option>
                        <option value="Cayman Islands">Cayman Islands</option>
                        <option value="Central African Republic">Central African Republic</option>
                        <option value="Chad">Chad</option>
                        <option value="Channel Islands">Channel Islands</option>
                        <option value="Chile">Chile</option>
                        <option value="China">China</option>
                        <option value="Christmas Island">Christmas Island</option>
                        <option value="Cocos Island">Cocos Island</option>
                        <option value="Colombia">Colombia</option>
                        <option value="Comoros">Comoros</option>
                        <option value="Congo">Congo</option>
                        <option value="Cook Islands">Cook Islands</option>
                        <option value="Costa Rica">Costa Rica</option>
                        <option value="Cote DIvoire">Cote DIvoire</option>
                        <option value="Croatia">Croatia</option>
                        <option value="Cuba">Cuba</option>
                        <option value="Curaco">Curacao</option>
                        <option value="Cyprus">Cyprus</option>
                        <option value="Czech Republic">Czech Republic</option>
                        <option value="Denmark">Denmark</option>
                        <option value="Djibouti">Djibouti</option>
                        <option value="Dominica">Dominica</option>
                        <option value="Dominican Republic">Dominican Republic</option>
                        <option value="East Timor">East Timor</option>
                        <option value="Ecuador">Ecuador</option>
                        <option value="Egypt">Egypt</option>
                        <option value="El Salvador">El Salvador</option>
                        <option value="Equatorial Guinea">Equatorial Guinea</option>
                        <option value="Eritrea">Eritrea</option>
                        <option value="Estonia">Estonia</option>
                        <option value="Ethiopia">Ethiopia</option>
                        <option value="Falkland Islands">Falkland Islands</option>
                        <option value="Faroe Islands">Faroe Islands</option>
                        <option value="Fiji">Fiji</option>
                        <option value="Finland">Finland</option>
                        <option value="France">France</option>
                        <option value="French Guiana">French Guiana</option>
                        <option value="French Polynesia">French Polynesia</option>
                        <option value="French Southern Ter">French Southern Ter</option>
                        <option value="Gabon">Gabon</option>
                        <option value="Gambia">Gambia</option>
                        <option value="Georgia">Georgia</option>
                        <option value="Germany">Germany</option>
                        <option value="Ghana">Ghana</option>
                        <option value="Gibraltar">Gibraltar</option>
                        <option value="Great Britain">Great Britain</option>
                        <option value="Greece">Greece</option>
                        <option value="Greenland">Greenland</option>
                        <option value="Grenada">Grenada</option>
                        <option value="Guadeloupe">Guadeloupe</option>
                        <option value="Guam">Guam</option>
                        <option value="Guatemala">Guatemala</option>
                        <option value="Guinea">Guinea</option>
                        <option value="Guyana">Guyana</option>
                        <option value="Haiti">Haiti</option>
                        <option value="Hawaii">Hawaii</option>
                        <option value="Honduras">Honduras</option>
                        <option value="Hong Kong">Hong Kong</option>
                        <option value="Hungary">Hungary</option>
                        <option value="Iceland">Iceland</option>
                        <option value="Indonesia">Indonesia</option>
                        <option value="India">India</option>
                        <option value="Iran">Iran</option>
                        <option value="Iraq">Iraq</option>
                        <option value="Ireland">Ireland</option>
                        <option value="Isle of Man">Isle of Man</option>
                        <option value="Israel">Israel</option>
                        <option value="Italy">Italy</option>
                        <option value="Jamaica">Jamaica</option>
                        <option value="Japan">Japan</option>
                        <option value="Jordan">Jordan</option>
                        <option value="Kazakhstan">Kazakhstan</option>
                        <option value="Kenya">Kenya</option>
                        <option value="Kiribati">Kiribati</option>
                        <option value="Korea North">Korea North</option>
                        <option value="Korea Sout">Korea South</option>
                        <option value="Kuwait">Kuwait</option>
                        <option value="Kyrgyzstan">Kyrgyzstan</option>
                        <option value="Laos">Laos</option>
                        <option value="Latvia">Latvia</option>
                        <option value="Lebanon">Lebanon</option>
                        <option value="Lesotho">Lesotho</option>
                        <option value="Liberia">Liberia</option>
                        <option value="Libya">Libya</option>
                        <option value="Liechtenstein">Liechtenstein</option>
                        <option value="Lithuania">Lithuania</option>
                        <option value="Luxembourg">Luxembourg</option>
                        <option value="Macau">Macau</option>
                        <option value="Macedonia">Macedonia</option>
                        <option value="Madagascar">Madagascar</option>
                        <option value="Malaysia">Malaysia</option>
                        <option value="Malawi">Malawi</option>
                        <option value="Maldives">Maldives</option>
                        <option value="Mali">Mali</option>
                        <option value="Malta">Malta</option>
                        <option value="Marshall Islands">Marshall Islands</option>
                        <option value="Martinique">Martinique</option>
                        <option value="Mauritania">Mauritania</option>
                        <option value="Mauritius">Mauritius</option>
                        <option value="Mayotte">Mayotte</option>
                        <option value="Mexico">Mexico</option>
                        <option value="Midway Islands">Midway Islands</option>
                        <option value="Moldova">Moldova</option>
                        <option value="Monaco">Monaco</option>
                        <option value="Mongolia">Mongolia</option>
                        <option value="Montserrat">Montserrat</option>
                        <option value="Morocco">Morocco</option>
                        <option value="Mozambique">Mozambique</option>
                        <option value="Myanmar">Myanmar</option>
                        <option value="Nambia">Nambia</option>
                        <option value="Nauru">Nauru</option>
                        <option value="Nepal">Nepal</option>
                        <option value="Netherland Antilles">Netherland Antilles</option>
                        <option value="Netherlands">Netherlands (Holland, Europe)</option>
                        <option value="Nevis">Nevis</option>
                        <option value="New Caledonia">New Caledonia</option>
                        <option value="New Zealand">New Zealand</option>
                        <option value="Nicaragua">Nicaragua</option>
                        <option value="Niger">Niger</option>
                        <option value="Nigeria">Nigeria</option>
                        <option value="Niue">Niue</option>
                        <option value="Norfolk Island">Norfolk Island</option>
                        <option value="Norway">Norway</option>
                        <option value="Oman">Oman</option>
                        <option value="Pakistan">Pakistan</option>
                        <option value="Palau Island">Palau Island</option>
                        <option value="Palestine">Palestine</option>
                        <option value="Panama">Panama</option>
                        <option value="Papua New Guinea">Papua New Guinea</option>
                        <option value="Paraguay">Paraguay</option>
                        <option value="Peru">Peru</option>
                        <option value="Phillipines">Philippines</option>
                        <option value="Pitcairn Island">Pitcairn Island</option>
                        <option value="Poland">Poland</option>
                        <option value="Portugal">Portugal</option>
                        <option value="Puerto Rico">Puerto Rico</option>
                        <option value="Qatar">Qatar</option>
                        <option value="Republic of Montenegro">Republic of Montenegro</option>
                        <option value="Republic of Serbia">Republic of Serbia</option>
                        <option value="Reunion">Reunion</option>
                        <option value="Romania">Romania</option>
                        <option value="Russia">Russia</option>
                        <option value="Rwanda">Rwanda</option>
                        <option value="St Barthelemy">St Barthelemy</option>
                        <option value="St Eustatius">St Eustatius</option>
                        <option value="St Helena">St Helena</option>
                        <option value="St Kitts-Nevis">St Kitts-Nevis</option>
                        <option value="St Lucia">St Lucia</option>
                        <option value="St Maarten">St Maarten</option>
                        <option value="St Pierre & Miquelon">St Pierre & Miquelon</option>
                        <option value="St Vincent & Grenadines">St Vincent & Grenadines</option>
                        <option value="Saipan">Saipan</option>
                        <option value="Samoa">Samoa</option>
                        <option value="Samoa American">Samoa American</option>
                        <option value="San Marino">San Marino</option>
                        <option value="Sao Tome & Principe">Sao Tome & Principe</option>
                        <option value="Saudi Arabia">Saudi Arabia</option>
                        <option value="Senegal">Senegal</option>
                        <option value="Seychelles">Seychelles</option>
                        <option value="Sierra Leone">Sierra Leone</option>
                        <option value="Singapore">Singapore</option>
                        <option value="Slovakia">Slovakia</option>
                        <option value="Slovenia">Slovenia</option>
                        <option value="Solomon Islands">Solomon Islands</option>
                        <option value="Somalia">Somalia</option>
                        <option value="South Africa">South Africa</option>
                        <option value="Spain">Spain</option>
                        <option value="Sri Lanka">Sri Lanka</option>
                        <option value="Sudan">Sudan</option>
                        <option value="Suriname">Suriname</option>
                        <option value="Swaziland">Swaziland</option>
                        <option value="Sweden">Sweden</option>
                        <option value="Switzerland">Switzerland</option>
                        <option value="Syria">Syria</option>
                        <option value="Tahiti">Tahiti</option>
                        <option value="Taiwan">Taiwan</option>
                        <option value="Tajikistan">Tajikistan</option>
                        <option value="Tanzania">Tanzania</option>
                        <option value="Thailand">Thailand</option>
                        <option value="Togo">Togo</option>
                        <option value="Tokelau">Tokelau</option>
                        <option value="Tonga">Tonga</option>
                        <option value="Trinidad & Tobago">Trinidad & Tobago</option>
                        <option value="Tunisia">Tunisia</option>
                        <option value="Turkey">Turkey</option>
                        <option value="Turkmenistan">Turkmenistan</option>
                        <option value="Turks & Caicos Is">Turks & Caicos Is</option>
                        <option value="Tuvalu">Tuvalu</option>
                        <option value="Uganda">Uganda</option>
                        <option value="United Kingdom">United Kingdom</option>
                        <option value="Ukraine">Ukraine</option>
                        <option value="United Arab Erimates">United Arab Emirates</option>
                        <option value="United States of America">United States of America</option>
                        <option value="Uraguay">Uruguay</option>
                        <option value="Uzbekistan">Uzbekistan</option>
                        <option value="Vanuatu">Vanuatu</option>
                        <option value="Vatican City State">Vatican City State</option>
                        <option value="Venezuela">Venezuela</option>
                        <option value="Vietnam">Vietnam</option>
                        <option value="Virgin Islands (Brit)">Virgin Islands (Brit)</option>
                        <option value="Virgin Islands (USA)">Virgin Islands (USA)</option>
                        <option value="Wake Island">Wake Island</option>
                        <option value="Wallis & Futana Is">Wallis & Futana Is</option>
                        <option value="Yemen">Yemen</option>
                        <option value="Zaire">Zaire</option>
                        <option value="Zambia">Zambia</option>
                        <option value="Zimbabwe">Zimbabwe</option>
                    </select>
                    <div class="invalid-feedback">Please select your country</div>
                </div>
                <div class="col-12 col-sm-12 col-md-6 col-lg-6 col-xl-6 col-xxl-6 mt-2">
                    <input class="form-control" type="text" name="state" placeholder="State" required>
                    <div class="invalid-feedback">Please enter your state</div>
                </div>
            </div>

            <div class="row mt-2">
                <div class="col-12 col-sm-12 col-md-6 col-lg-6 col-xl-6 col-xxl-6 mt-2">
                    <input class="form-control" type="text" name="city" placeholder="City" required>
                    <div class="invalid-feedback">Please enter your city</div>
                </div>
                <div class="col-12 col-sm-12 col-md-6 col-lg-6 col-xl-6 col-xxl-6 mt-2">
                    <input class="form-control" type="text" name="zipcode" placeholder="Zip Code" required>
                    <div class="invalid-feedback">Please enter your zip code</div>
                </div>
            </div>
            <div class="row">
                <p class="primary-text mt-4">2. Contact Person</p>
                <div class="row">
                    <div class="col-12 col-sm-12 col-md-6 col-lg-6 col-xl-6 col-xxl-6 mt-2">
                        <input class="form-control" type="text" name="fullname" placeholder="Full Name" required>
                        <div class="invalid-feedback">Please enter your full name</div>
                    </div>
                    <div class="col-12 col-sm-12 col-md-6 col-lg-6 col-xl-6 col-xxl-6 mt-2">
                        <input class="form-control" type="text" name="titlerole" placeholder="Title/Role" required>
                        <div class="invalid-feedback">Please enter your title role</div>
                    </div>
                </div>
                <div class="row mt-2">
                    <div class="col-12 col-sm-12 col-md-6 col-lg-6 col-xl-6 col-xxl-6 mt-2">
                        <input class="form-control" type="text" name="phone" placeholder="Phone" required>
                        <div class="invalid-feedback">Please enter your phone</div>
                    </div>
                    <div class="col-12 col-sm-12 col-md-6 col-lg-6 col-xl-6 col-xxl-6 mt-2">
                        <input class="form-control" type="email" name="email" placeholder="Email" required>
                        <div class="invalid-feedback">Please enter your email</div>
                    </div>
                </div>
            </div>

            <div class="row">
                <p class="primary-text mt-4">3. Partnership Interests</p>
                <p class="mt-2">What areas of collaboration are you interested in? (Check all that apply)</p>
                <div class="col-12 col-sm-12 col-md-12 col-lg-12 col-xl-12 col-xxl-12">
                    <div class="form-check collaboration-areas-group">
                        <div>
                            <input class="form-check-input" type="checkbox" name="collaboration_areas[]" value="Event Sponsorship" id="eventsponsorship">
                            <label class="form-check-label" for="eventsponsorship">Event Sponsorship</label>
                        </div>
                        <div>
                            <input class="form-check-input" type="checkbox" name="collaboration_areas[]" value="Cultural Programming" id="culturalprogramming">
                            <label class="form-check-label" for="culturalprogramming">Cultural Programming</label>
                        </div>
                        <div>
                            <input class="form-check-input" type="checkbox" name="collaboration_areas[]" value="Language & Education Initiatives" id="languageeducationinitiatives">
                            <label class="form-check-label" for="languageeducationinitiatives">Language & Education Initiatives</label>
                        </div>
                        
                        <div>
                            <input class="form-check-input" type="checkbox" name="collaboration_areas[]" value="Youth Engagement" id="youthengagement">
                            <label class="form-check-label" for="youthengagement">Youth Engagement</label>
                        </div>
                        <div>
                            <input class="form-check-input" type="checkbox" name="collaboration_areas[]" value="Women’s Empowerment Programs" id="womenempowermentprograms">
                            <label class="form-check-label" for="womenempowermentprograms">Women’s Empowerment Programs</label>
                        </div>
                        <div>
                            <input class="form-check-input" type="checkbox" name="collaboration_areas[]" value="Community Outreach" id="communityoutreach">
                            <label class="form-check-label" for="communityoutreach">Community Outreach</label>
                        </div>

                        <div>
                            <input class="form-check-input" type="checkbox" name="collaboration_areas[]" value="In-Kind Support or Donations" id="inkindsupportordonations">
                            <label class="form-check-label" for="inkindsupportordonations">In-Kind Support or Donations</label>
                        </div>

                    </div>
                    <div>
                        <label for="other_collab"><?php _e('Other (please describe):', 'bootstrap-partner-inquiry'); ?></label>
                        <textarea id="other_collab" name="other_collaboration_areas" class="form-control"></textarea>
                    </div>
                    <div class="invalid-feedback collaboration-areas-feedback">Please check at least 1 partnership interests or fill other textarea</div>
                </div>
            </div>

            <div class="row">
                <p class="primary-text mt-4">4. <?php _e('Tell Us More', 'bootstrap-partner-inquiry'); ?></p>
                <p class="mb-0 mt-2"><?php _e('Why would you like to partner with Shako Mako?', 'bootstrap-partner-inquiry'); ?></p>
                <div>
                    <textarea class="form-control" name="why_partner" id="wouldyouliketopartner" required></textarea>
                    <div class="invalid-feedback"><?php _e('Please tell us why you\'d like to partner.', 'bootstrap-partner-inquiry'); ?></div>
                </div>
                
                <p class="mb-0 mt-2"><?php _e('Do you have a specific project, program, or idea in mind?', 'bootstrap-partner-inquiry'); ?></p>
                <div class="row">
                    <div class="col-12 col-sm-12 col-md-12 col-lg-12 col-xl-12 col-xxl-12 mt-2">
                        <div class="form-check specific-project-group">
                            <div>
                                <input class="form-check-input circle" type="radio" name="specific_project_idea" value="Yes" id="specific_project_yes" required>
                                <label class="form-check-label" for="specific_project_yes"><?php _e('Yes', 'bootstrap-partner-inquiry'); ?></label>
                            </div>
                            <div>
                                <input class="form-check-input circle" type="radio" name="specific_project_idea" value="No" checked id="specific_project_no">
                                <label class="form-check-label" for="specific_project_no"><?php _e('No', 'bootstrap-partner-inquiry'); ?></label>
                            </div>
                        </div>
                        <div class="invalid-feedback specific-project-feedback"><?php _e('Please select Yes or No.', 'bootstrap-partner-inquiry'); ?></div>
                    </div>
                    <p class="mb-0 mt-2" id="specific_project_desc_label" style="display:none;"><?php _e('If yes, please describe briefly:', 'bootstrap-partner-inquiry'); ?></p>
                    <div>
                        <textarea class="form-control" name="specific_project_description" id="specific_project_description" style="display:none;"></textarea>
                        <div class="invalid-feedback specific-project-description-feedback" style="display:none;"><?php _e('Please describe your specific project, program, or idea.', 'bootstrap-partner-inquiry'); ?></div>
                    </div>
                </div>

                <div class="row">
                    <p class="primary-text mt-4">5. <?php _e('Additional Information (Optional)', 'bootstrap-partner-inquiry'); ?></p>
                    <p class="mb-0"><?php _e('Is there anything else you’d like us to know about your organization or proposal?', 'bootstrap-partner-inquiry'); ?></p>
                    <div>
                        <textarea class="form-control" name="additional_info"></textarea>
                    </div>
                </div>

                <div class="row">
                    <p class="primary-text mt-4">6. <?php _e('Follow-Up Preferences', 'bootstrap-partner-inquiry'); ?></p>
                    <div class="col-12 col-sm-12 col-md-12 col-lg-12 col-xl-12 col-xxl-12 mt-2">
                        <div>
                            <input class="form-check-input" type="checkbox" name="followup_schedule_meeting" value="Yes" id="followup_schedule_meeting">
                            <label class="form-check-label long" for="followup_schedule_meeting"><?php _e('I would like to schedule a meeting to discuss potential partnership opportunities', 'bootstrap-partner-inquiry'); ?></label>
                        </div>
                    </div>
                    <div class="col-12 col-sm-12 col-md-12 col-lg-12 col-xl-12 col-xxl-12 mt-2">
                        <div>
                            <input class="form-check-input" type="checkbox" name="followup_send_packet" value="Yes" id="followup_send_packet">
                            <label class="form-check-label long" for="followup_send_packet"><?php _e('Please send me your partnership and sponsorship packet', 'bootstrap-partner-inquiry'); ?></label>
                        </div>
                    </div>
                    <div class="col-12 col-sm-12 col-md-12 col-lg-12 col-xl-12 col-xxl-12 mt-2">
                        <div>
                            <input class="form-check-input" type="checkbox" name="followup_mailing_list" value="Yes" id="followup_mailing_list">
                            <label class="form-check-label long" for="followup_mailing_list"><?php _e('Add me to your mailing list for updates and upcoming events', 'bootstrap-partner-inquiry'); ?></label>
                        </div>
                    </div>
                </div>

                <p class="mt-4"><?php _e('Thank you for reaching out! We look forward to the opportunity to collaborate and make an impact together. For questions, contact us at', 'bootstrap-partner-inquiry'); ?> <a href="mailto:partnerships@shakomakoiaa.org" style="color: #FFA300;">partnerships@shakomakoiaa.org</a></p>

                <div class="row justify-content-center mt-3">
                    <button class="btn btn-primary w-10" type="submit" name="submit_partner_form"><?php _e('Submit', 'bootstrap-partner-inquiry'); ?></button>
                </div>
            <div class="bpi-form-messages text-center m-3">
                <!-- AJAX messages will be displayed here -->
            </div>
            <?php wp_nonce_field( 'partner_form_nonce', 'partner_form_nonce_field' ); ?>
            </form>
        </div>
    </div>
    <?php
    return ob_get_clean(); // Return the buffered content
}

/**
 * Handle AJAX form submission for the partner inquiry form.
 * This function processes the submitted data, saves it to the 'partner' CPT
 * using ACF, and sends email notifications.
 */
function bpi_handle_ajax_form_submission() {
    // Verify nonce for security.
    if ( ! isset( $_POST['partner_form_nonce_field'] ) || ! wp_verify_nonce( $_POST['partner_form_nonce_field'], 'partner_form_nonce' ) ) {
        wp_send_json_error( array( 'message' => __( 'Security check failed. Please try again.', 'bootstrap-partner-inquiry' ) ) );
        wp_die();
    }

    // Check if ACF is active. This plugin relies heavily on ACF.
    if ( ! function_exists( 'update_field' ) ) {
        wp_send_json_error( array( 'message' => __( 'Advanced Custom Fields plugin is not active. Please install and activate it.', 'bootstrap-partner-inquiry' ) ) );
        wp_die();
    }

    // Sanitize and validate inputs.
    $organization_name          = sanitize_text_field( $_POST['organizationname'] );
    $type_of_organization       = isset( $_POST['type_of_organization'] ) ? array_map( 'sanitize_text_field', (array) $_POST['type_of_organization'] ) : array();
    $website_social_media       = sanitize_text_field( $_POST['websiteorsocialmedia'] );
    $country                    = sanitize_text_field( $_POST['country'] );
    $state                      = sanitize_text_field( $_POST['state'] );
    $city                       = sanitize_text_field( $_POST['city'] );
    $zipcode                    = sanitize_text_field( $_POST['zipcode'] );
    $contact_fullname           = sanitize_text_field( $_POST['fullname'] );
    $contact_title_role         = sanitize_text_field( $_POST['titlerole'] );
    $contact_phone              = sanitize_text_field( $_POST['phone'] );
    $contact_email              = sanitize_email( $_POST['email'] );
    $why_partner                = sanitize_textarea_field( $_POST['why_partner'] );
    $collaboration_areas        = isset( $_POST['collaboration_areas'] ) ? array_map( 'sanitize_text_field', (array) $_POST['collaboration_areas'] ) : array();
    $other_collaboration_areas  = sanitize_textarea_field( $_POST['other_collaboration_areas'] );
    $specific_project_idea      = sanitize_text_field( $_POST['specific_project_idea'] );
    $specific_project_description = sanitize_textarea_field( $_POST['specific_project_description'] );
    $additional_info            = sanitize_textarea_field( $_POST['additional_info'] );
    // Ensure optional checkboxes save 'Yes' or 'No'
    $followup_schedule_meeting  = isset($_POST['followup_schedule_meeting']) ? 'Yes' : 'No';
    $followup_send_packet       = isset($_POST['followup_send_packet']) ? 'Yes' : 'No';
    $followup_mailing_list      = isset($_POST['followup_mailing_list']) ? 'Yes' : 'No';

    // Server-side validation
    if ( empty( $organization_name ) || empty( $country ) || empty( $state ) || empty( $city ) || empty( $zipcode ) || empty( $contact_fullname ) || empty( $contact_title_role ) || empty( $contact_phone ) || ! is_email( $contact_email ) || empty( $why_partner ) || empty( $specific_project_idea ) ) {
        wp_send_json_error( array( 'message' => __( 'Please fill in all required fields correctly.', 'bootstrap-partner-inquiry' ) ) );
        wp_die();
    }

    // Validation for "Type of Organization" checkboxes
    if ( empty( $type_of_organization ) ) {
        wp_send_json_error( array( 'message' => __( 'Please select at least one type of organization.', 'bootstrap-partner-inquiry' ) ) );
        wp_die();
    }

    // Validation for "Areas of Collaboration" checkboxes or "Other" textarea
    if ( empty( $collaboration_areas ) && empty( $other_collaboration_areas ) ) {
        wp_send_json_error( array( 'message' => __( 'Please select at least one area of collaboration or describe in "Other".', 'bootstrap-partner-inquiry' ) ) ) ;
        wp_die();
    }

    // Conditional validation for specific project description
    if ( $specific_project_idea === 'Yes' && empty( $specific_project_description ) ) {
        wp_send_json_error( array( 'message' => __( 'Please describe your specific project, program, or idea.', 'bootstrap-partner-inquiry' ) ) ) ;
        wp_die();
    }

    // Prepare post data for the new custom post type entry.
    $post_data = array(
        'post_title'    => sprintf( __( 'Inquiry from %s - %s', 'bootstrap-partner-inquiry' ), $organization_name, $contact_fullname ),
        'post_status'   => 'publish',
        'post_type'     => 'partner',
    );

    // Insert the new post into the WordPress database.
    $post_id = wp_insert_post( $post_data );

    if ( ! is_wp_error( $post_id ) ) {
        // Save custom fields using Advanced Custom Fields (ACF).
        update_field( 'partners_organization_name', $organization_name, $post_id );
        update_field( 'partners_type_of_organization', $type_of_organization, $post_id ); // Checkbox field
        update_field( 'partners_website_social_media', $website_social_media, $post_id );
        update_field( 'partners_country', $country, $post_id );
        update_field( 'partners_state', $state, $post_id );
        update_field( 'partners_city', $city, $post_id );
        update_field( 'partners_zipcode', $zipcode, $post_id );
        update_field( 'partners_contact_fullname', $contact_fullname, $post_id );
        update_field( 'partners_contact_title_role', $contact_title_role, $post_id );
        update_field( 'partners_contact_phone', $contact_phone, $post_id );
        update_field( 'partners_contact_email', $contact_email, $post_id );
        update_field( 'partners_why_partner', $why_partner, $post_id );
        update_field( 'partners_collaboration_areas', $collaboration_areas, $post_id ); // Checkbox field
        update_field( 'partners_other_collaboration_areas', $other_collaboration_areas, $post_id );
        update_field( 'partners_specific_project_idea', $specific_project_idea, $post_id ); // Radio button
        update_field( 'partners_specific_project_description', $specific_project_description, $post_id );
        update_field( 'partners_additional_info', $additional_info, $post_id );
        update_field( 'partners_followup_schedule_meeting', $followup_schedule_meeting, $post_id ); // Checkbox
        update_field( 'partners_followup_send_packet', $followup_send_packet, $post_id ); // Checkbox
        update_field( 'partners_followup_mailing_list', $followup_mailing_list, $post_id ); // Checkbox

        // Retrieve admin email from WordPress options
        $admin_email_recipient = get_option( 'bpi_admin_email_recipient', get_option( 'admin_email' ) ); // Fallback to WP admin email
        $site_name   = get_bloginfo( 'name' );

        /**
         * Send Admin Notification Email
         */
        $to_admin = $admin_email_recipient;
        $subject_admin = sprintf( __( 'New Partner Inquiry from %s (%s)', 'bootstrap-partner-inquiry' ), $organization_name, $contact_fullname );
        $message_admin = '<p>' . __( 'You have received a new partner inquiry:', 'bootstrap-partner-inquiry' ) . '</p>';
        $message_admin .= '<ul>';
        $message_admin .= '<li><strong>' . __( 'Organization Name:', 'bootstrap-partner-inquiry' ) . '</strong> ' . esc_html( $organization_name ) . '</li>';
        $message_admin .= '<li><strong>' . __( 'Contact Name:', 'bootstrap-partner-inquiry' ) . '</strong> ' . esc_html( $contact_fullname ) . '</li>';
        $message_admin .= '<li><strong>' . __( 'Contact Email:', 'bootstrap-partner-inquiry' ) . '</strong> ' . esc_html( $contact_email ) . '</li>';
        $message_admin .= '<li><strong>' . __( 'Country:', 'bootstrap-partner-inquiry' ) . '</strong> ' . esc_html( $country ) . '</li>';
        $message_admin .= '<li><strong>' . __( 'Areas of Collaboration:', 'bootstrap-partner-inquiry' ) . '</strong> ' . esc_html( implode(', ', $collaboration_areas) . (!empty($other_collaboration_areas) ? ' (' . $other_collaboration_areas . ')' : '') ) . '</li>';
        $message_admin .= '</ul>';
        $message_admin .= '<p>' . sprintf( __( 'View the full inquiry in WordPress admin: %s', 'bootstrap-partner-inquiry' ), get_edit_post_link( $post_id ) ) . '</p>';
        $headers_admin = array( 'Content-Type: text/html; charset=UTF-8', 'From: ' . $site_name . ' <' . $admin_email_recipient . '>' );
        $mail_sent_admin = wp_mail( $to_admin, $subject_admin, $message_admin, $headers_admin );

        /**
         * Send User Confirmation Email
         */
        $to_user = $contact_email;
        $subject_user = sprintf( __( 'Thank you for your Partner Inquiry to %s', 'bootstrap-partner-inquiry' ), $site_name );
        $message_user = '<p>' . sprintf( __( 'Hello %s,', 'bootstrap-partner-inquiry' ), esc_html( $contact_fullname ) ) . '</p>';
        $message_user .= '<p>' . __( 'Thank you for your interest in partnering with us! We have received your inquiry and will review it shortly.', 'bootstrap-partner-inquiry' ) . '</p>';
        $message_user .= '<p>' . __( 'Here is a summary of your submission:', 'bootstrap-partner-inquiry' ) . '</p>';
        $message_user .= '<ul>';
        $message_user .= '<li><strong>' . __( 'Organization:', 'bootstrap-partner-inquiry' ) . '</strong> ' . esc_html( $organization_name ) . '</li>';
        $message_user .= '<li><strong>' . __( 'Subject:', 'bootstrap-partner-inquiry' ) . '</strong> ' . esc_html( implode(', ', $collaboration_areas) . (!empty($other_collaboration_areas) ? ' (' . $other_collaboration_areas . ')' : '') ) . '</li>';
        $message_user .= '<li><strong>' . __( 'Your Message:', 'bootstrap-partner-inquiry' ) . '</strong><br>' . wp_kses_post( nl2br( $why_partner ) ) . '</li>';
        $message_user .= '</ul>';
        $message_user .= '<p>' . sprintf( __( 'Best regards,<br>%s Team', 'bootstrap-partner-inquiry' ), $site_name ) . '</p>';
        $headers_user = array( 'Content-Type: text/html; charset=UTF-8', 'From: ' . $site_name . ' <' . $admin_email_recipient . '>' );
        $mail_sent_user = wp_mail( $to_user, $subject_user, $message_user, $headers_user );

        // Determine final response based on email sending status
        if ( ! $mail_sent_admin || ! $mail_sent_user ) {
            wp_send_json_success( array( 'message' => __( 'Your inquiry was saved, but there was an issue sending one or more confirmation emails. Please contact support.', 'bootstrap-partner-inquiry' ), 'type' => 'warning' ) );
        } else {
            wp_send_json_success( array( 'message' => __( 'Thank you! Your inquiry has been submitted successfully.', 'bootstrap-partner-inquiry' ), 'type' => 'success' ) );
        }

    } else {
        wp_send_json_error( array( 'message' => __( 'There was an error saving your inquiry. Please try again.', 'bootstrap-partner-inquiry' ) ) );
    }

    wp_die(); // Always die in AJAX handlers.
}


/**
 * Add custom columns to the 'Partner Inquiries' list table in the WordPress admin.
 */
add_filter('manage_partner_posts_columns', 'bpi_add_partner_columns');
function bpi_add_partner_columns( $columns ) {
    $new_columns = array(
        'partners_contact_fullname' => __( 'Contact Person', 'bootstrap-partner-inquiry' ),
        'partners_contact_email'    => __( 'Email', 'bootstrap-partner-inquiry' ),
        'partners_country'          => __( 'Country', 'bootstrap-partner-inquiry' ),
    );

    // Insert new columns after the 'title' column (organization name).
    $offset = array_search( 'title', array_keys( $columns ) ) + 1;
    $columns = array_slice( $columns, 0, $offset, true ) + $new_columns + array_slice( $columns, $offset, null, true );

    return $columns;
}

/**
 * Populate custom columns in the 'Partner Inquiries' list table.
 */
add_action('manage_partner_posts_custom_column', 'bpi_populate_partner_columns', 10, 2);
function bpi_populate_partner_columns( $column, $post_id ) {
    // Ensure ACF is active before trying to get fields.
    if ( ! function_exists( 'get_field' ) ) {
        echo __( 'ACF not active', 'bootstrap-partner-inquiry' );
        return;
    }

    switch ( $column ) {
        case 'partners_contact_fullname':
            echo esc_html( get_field( 'partners_contact_fullname', $post_id ) );
            break;
        case 'partners_contact_email':
            echo esc_html( get_field( 'partners_contact_email', $post_id ) );
            break;
        case 'partners_country':
            echo esc_html( get_field( 'partners_country', $post_id ) );
            break;
    }
}

/**
 * Add new submenu pages under the 'Partner Inquiries' CPT.
 */
add_action('admin_menu', 'bpi_add_settings_pages');
function bpi_add_settings_pages() {
    // Add Email Settings page
    add_submenu_page(
        'edit.php?post_type=partner', // Parent slug (our custom post type)
        __('Partner Settings', 'bootstrap-partner-inquiry'), // Page title
        __('Email Settings', 'bootstrap-partner-inquiry'), // Menu title
        'manage_options', // Capability required to access the page
        'bpi-settings', // Menu slug
        'bpi_settings_page_content' // Callback function to display the page content
    );

    // Add How To Use page
    add_submenu_page(
        'edit.php?post_type=partner', // Parent slug (our custom post type)
        __('How To Use Partner Form', 'bootstrap-partner-inquiry'), // Page title
        __('How To Use', 'bootstrap-partner-inquiry'), // Menu title
        'manage_options', // Capability required to access the page
        'bpi-how-to-use', // Menu slug
        'bpi_how_to_use_page_content' // Callback function to display the page content
    );
}

/**
 * Register settings for the plugin's admin page.
 */
add_action('admin_init', 'bpi_register_settings');
function bpi_register_settings() {
    // Register a new setting for our plugin
    register_setting(
        'bpi_settings_group', // Option group
        'bpi_admin_email_recipient', // Option name (this is where the email will be stored)
        'sanitize_email' // Sanitize callback function
    );

    // Add a settings section for email settings
    add_settings_section(
        'bpi_email_settings_section', // ID
        __('Admin Email Notification', 'bootstrap-partner-inquiry'), // Title
        'bpi_email_settings_section_callback', // Callback for section content
        'bpi-settings' // Page slug
    );

    // Add a settings field
    add_settings_field(
        'bpi_admin_email_field', // ID
        __('Recipient Email', 'bootstrap-partner-inquiry'), // Title
        'bpi_admin_email_field_callback', // Callback for field content
        'bpi-settings', // Page slug
        'bpi_email_settings_section', // Section ID
        array(
            'label_for' => 'bpi_admin_email_recipient',
            'class'     => 'bpi-admin-email-row',
            'name'      => 'bpi_admin_email_recipient',
        )
    );
}

/**
 * Callback for the email settings section content.
 */
function bpi_email_settings_section_callback() {
    echo '<p>' . __('Enter the email address where you want to receive partner inquiry submissions.', 'bootstrap-partner-inquiry') . '</p>';
}

/**
 * Callback for the email settings field content.
 */
function bpi_admin_email_field_callback($args) {
    // Get the current value of the option, with a fallback to the WordPress admin email.
    $email = get_option($args['name'], get_option('admin_email'));
    ?>
    <input type="email" id="<?php echo esc_attr($args['name']); ?>" name="<?php echo esc_attr($args['name']); ?>" value="<?php echo esc_attr($email); ?>" class="regular-text" placeholder="<?php esc_attr_e('e.g., your-email@example.com', 'bootstrap-partner-inquiry'); ?>" />
    <p class="description">
        <?php _e('This email address will receive all new partner inquiry submissions. If left empty, the default WordPress admin email will be used.', 'bootstrap-partner-inquiry'); ?>
    </p>
    <?php
}

/**
 * Display the content of the plugin's Email Settings page.
 */
function bpi_settings_page_content() {
    ?>
    <div class="wrap">
        <h1><?php _e('Bootstrap Partner Inquiry Settings', 'bootstrap-partner-inquiry'); ?></h1>
        <form action="options.php" method="post">
            <?php
            settings_fields('bpi_settings_group'); // Output security fields for the settings group
            do_settings_sections('bpi-settings'); // Output settings sections and fields
            submit_button(); // Output save button
            ?>
        </form>
    </div>
    <?php
}

/**
 * Display the content of the plugin's "How To Use" page.
 */
function bpi_how_to_use_page_content() {
    ?>
    <div class="wrap">
        <h1><?php _e('How To Use Bootstrap Partner Inquiry Form', 'bootstrap-partner-inquiry'); ?></h1>
        <p><?php _e('To display the partner inquiry form on any page or post, simply use the following shortcode:', 'bootstrap-partner-inquiry'); ?></p>
        <pre><code>[partner_inquiry_form]</code></pre>
        <p><?php _e('Paste this shortcode into the content editor of the desired page or post.', 'bootstrap-partner-inquiry'); ?></p>
        
        <h2><?php _e('Required Advanced Custom Fields (ACF) Setup', 'bootstrap-partner-inquiry'); ?></h2>
        <p><?php _e('This plugin relies on the Advanced Custom Fields (ACF) plugin to store partner inquiry data. Please ensure ACF is installed and active.', 'bootstrap-partner-inquiry'); ?></p>
        <p><?php _e('You need to create a Field Group in ACF and assign it to the "Partner Inquiry" Custom Post Type. Below are the exact field names (keys) and recommended types you should use:', 'bootstrap-partner-inquiry'); ?></p>
        
        <table class="wp-list-table widefat fixed striped">
            <thead>
                <tr>
                    <th><?php _e('ACF Field Name (Key)', 'bootstrap-partner-inquiry'); ?></th>
                    <th><?php _e('Recommended Field Type', 'bootstrap-partner-inquiry'); ?></th>
                    <th><?php _e('Description', 'bootstrap-partner-inquiry'); ?></th>
                </tr>
            </thead>
            <tbody>
                <tr><td><code>partners_organization_name</code></td><td>Text</td><td>Name of the organization.</td></tr>
                <tr><td><code>partners_type_of_organization</code></td><td>Checkbox</td><td>Type of organization (e.g., Nonprofit, Business).</td></tr>
                <tr><td><code>partners_website_social_media</code></td><td>URL / Text</td><td>Organization's website or social media links.</td></tr>
                <tr><td><code>partners_country</code></td><td>Select</td><td>Organization's country.</td></tr>
                <tr><td><code>partners_state</code></td><td>Text</td><td>Organization's state.</td></tr>
                <tr><td><code>partners_city</code></td><td>Text</td><td>Organization's city.</td></tr>
                <tr><td><code>partners_zipcode</code></td><td>Text</td><td>Organization's zip code.</td></tr>
                <tr><td><code>partners_contact_fullname</code></td><td>Text</td><td>Full name of the contact person.</td></tr>
                <tr><td><code>partners_contact_title_role</code></td><td>Text</td><td>Title/Role of the contact person.</td></tr>
                <tr><td><code>partners_contact_phone</code></td><td>Text</td><td>Phone number of the contact person.</td></tr>
                <tr><td><code>partners_contact_email</code></td><td>Email</td><td>Email of the contact person.</td></tr>
                <tr><td><code>partners_why_partner</code></td><td>Text Area</td><td>Reason for wanting to partner.</td></tr>
                <tr><td><code>partners_collaboration_areas</code></td><td>Checkbox</td><td>Areas of collaboration interested in.</td></tr>
                <tr><td><code>partners_other_collaboration_areas</code></td><td>Text Area</td><td>Description of other collaboration areas.</td></tr>
                <tr><td><code>partners_specific_project_idea</code></td><td>Radio Button</td><td>Does the organization have a specific project/idea? (Yes/No)</td></tr>
                <tr><td><code>partners_specific_project_description</code></td><td>Text Area</td><td>Description of the specific project/idea (conditionally required).</td></tr>
                <tr><td><code>partners_additional_info</code></td><td>Text Area</td><td>Any additional information about the organization or proposal.</td></tr>
                <tr><td><code>partners_followup_schedule_meeting</code></td><td>Text / Checkbox</td><td>Preference to schedule a meeting (saved as 'Yes' or 'No').</td></tr>
                <tr><td><code>partners_followup_send_packet</code></td><td>Text / Checkbox</td><td>Preference to receive partnership packet (saved as 'Yes' or 'No').</td></tr>
                <tr><td><code>partners_followup_mailing_list</code></td><td>Text / Checkbox</td><td>Preference to be added to mailing list (saved as 'Yes' or 'No').</td></tr>
            </tbody>
        </table>
        
        <h2><?php _e('Email Notification Settings', 'bootstrap-partner-inquiry'); ?></h2>
        <p><?php _e('You can configure the recipient email address for partner inquiry notifications under the "Email Settings" submenu item.', 'bootstrap-partner-inquiry'); ?></p>
    </div>
    <?php
}

/**
 * Add a dashboard widget to display the latest partner inquiries.
 */
add_action('wp_dashboard_setup', 'bpi_add_dashboard_widgets');
function bpi_add_dashboard_widgets() {
    // Check if ACF is active before adding the widget.
    if ( ! function_exists( 'get_field' ) ) {
        return; // Do not add the widget if ACF is not active.
    }

    // Query to check if there are any partner inquiries.
    $check_inquiries_query = new WP_Query( array(
        'post_type'      => 'partner',
        'posts_per_page' => 1, // Only need to check for one post to know if any exist.
        'post_status'    => 'publish',
        'fields'         => 'ids', // Only retrieve post IDs for efficiency.
    ) );

    if ( $check_inquiries_query->have_posts() ) {
        // If there are inquiries, add the dashboard widget.
        wp_add_dashboard_widget(
            'bpi_latest_partner_inquiries',         // Widget slug.
            __( 'Latest Partner Inquiries', 'bootstrap-partner-inquiry' ), // Title.
            'bpi_latest_partner_inquiries_callback' // Display function.
        );
    }
    // No need for an else block; if no posts, the widget simply isn't added.
}

/**
 * Callback function to display the content of the dashboard widget.
 */
function bpi_latest_partner_inquiries_callback() {
    // Check if ACF is active before trying to get fields.
    if ( ! function_exists( 'get_field' ) ) {
        echo '<p>' . __( 'Advanced Custom Fields plugin is not active. Please install and activate it to view partner inquiries here.', 'bootstrap-partner-inquiry' ) . '</p>';
        return;
    }

    // The number of items to show is now fixed at 3.
    $num_items = 3; 

    // Query the latest partner inquiries.
    $args = array(
        'post_type'      => 'partner',
        'posts_per_page' => $num_items,
        'post_status'    => 'publish',
        'orderby'        => 'date',
        'order'          => 'DESC', // Latest to oldest
    );

    $partner_inquiries = new WP_Query( $args );

    if ( $partner_inquiries->have_posts() ) {
        echo '<ul class="bpi-dashboard-inquiries">';
        while ( $partner_inquiries->have_posts() ) {
            $partner_inquiries->the_post();
            $post_id = get_the_ID();
            $contact_fullname = get_field( 'partners_contact_fullname', $post_id );
            $contact_email = get_field( 'partners_contact_email', $post_id );
            $country = get_field( 'partners_country', $post_id );
            $edit_post_link = get_edit_post_link( $post_id );

            echo '<li>';
            echo '<strong>' . esc_html( $contact_fullname ) . '</strong>';
            echo ' (' . esc_html( $contact_email ) . ')';
            echo '<br><strong>' . __( 'Country:', 'bootstrap-partner-inquiry' ) . '</strong> ' . esc_html( $country );
            echo '<br><br><a href="' . esc_url( $edit_post_link ) . '" class="button button-primary button-small">' . __( 'View Inquiry', 'bootstrap-partner-inquiry' ) . '</a>';
            echo '<hr></li>';
        }
        echo '</ul>';
        wp_reset_postdata(); // Restore original Post Data
    } else {
        // This block will technically not be reached if the widget is only added when posts exist,
        // but it's good practice to have a fallback message if somehow it does.
        echo '<p>' . __( 'No partner inquiries found.', 'bootstrap-partner-inquiry' ) . '</p>';
    }
}