/**
 * Front-end JavaScript for Bootstrap Partner Inquiry plugin.
 * Handles AJAX form submission, client-side validation, and conditional logic.
 */

jQuery(document).ready(function($) {
	const $form = $('.bootstrap-partner-form__form'); // Target the form using its class
	const $messagesDiv = $form.find('.bpi-form-messages'); // Div for displaying AJAX messages

	// Elements for conditional display
	const $specificProjectYesRadio = $('#specific_project_yes');
	const $specificProjectDescription = $('#specific_project_description');
	const $specificProjectDescLabel = $('#specific_project_desc_label');
	const $specificProjectDescriptionFeedback = $form.find('.specific-project-description-feedback');

	/**
	 * Conditional display for "If yes, please describe briefly" textarea.
	 */
	function toggleSpecificProjectDescription() {
		if($specificProjectYesRadio.is(':checked')) {
			$specificProjectDescription.show().attr('required', 'required');
			$specificProjectDescLabel.show();
			$specificProjectDescriptionFeedback.show();
		} else {
			$specificProjectDescription.hide().removeAttr('required').val(''); // Hide and clear
			$specificProjectDescLabel.hide();
			$specificProjectDescription.removeClass('is-invalid').siblings('.invalid-feedback').hide(); // Clear validation
			$specificProjectDescriptionFeedback.hide();
		}
	}

	// Attach event listeners for the radio buttons
	$form.find('input[name="specific_project_idea"]').on('change', toggleSpecificProjectDescription);

	// Initial check on page load (in case form is pre-filled or cached)
	toggleSpecificProjectDescription();

	/**
	 * Custom Client-Side Validation Logic and AJAX Form Submission
	 */
	$(document).on('submit', '.bootstrap-partner-form__form', function(e) {
		e.preventDefault(); // Prevent the default browser form submission.
		e.stopPropagation(); // Stop event propagation.

		const $form = $(this);
		const form = $form[0]; // Get the native DOM form element for `checkValidity()`.
		const $messagesDiv = $form.find('.bpi-form-messages'); // Message div within THIS form.
		const $submitButton = $form.find('button[type="submit"]');
		const originalButtonText = $submitButton.html(); // Store original button text.

		// Clear any previous messages and remove validation classes.
		$messagesDiv.empty().removeClass('alert alert-success alert-warning alert-danger');
		$form.removeClass('was-validated'); // Remove Bootstrap's validation class.

		let formIsValid = true; // Flag to track overall form validity.

		// --- 1. Native HTML5 Validation Check ---
		if(!form.checkValidity()) {
			formIsValid = false;
		}

		// --- 2. Custom Checkbox Validation: Type of Organization ---
		const $orgTypeCheckboxes = $form.find('input[name="type_of_organization[]"]');
		const $orgTypeFeedback = $form.find('.organization-type-feedback');
		if($orgTypeCheckboxes.filter(':checked').length === 0) {
			$orgTypeCheckboxes.addClass('is-invalid');
			$orgTypeFeedback.show();
			formIsValid = false;
		} else {
			$orgTypeCheckboxes.removeClass('is-invalid');
			$orgTypeFeedback.hide();
		}

		// --- 3. Custom Checkbox/Textarea Validation: Collaboration Areas ---
		const $collabAreaCheckboxes = $form.find('input[name="collaboration_areas[]"]');
		const $otherCollabTextarea = $form.find('textarea[name="other_collaboration_areas"]');
		const $collabAreaFeedback = $form.find('.collaboration-areas-feedback');

		if($collabAreaCheckboxes.filter(':checked').length === 0 && $otherCollabTextarea.val().trim() === '') {
			$collabAreaCheckboxes.addClass('is-invalid');
			$otherCollabTextarea.addClass('is-invalid');
			$collabAreaFeedback.show();
			formIsValid = false;
		} else {
			$collabAreaCheckboxes.removeClass('is-invalid');
			$otherCollabTextarea.removeClass('is-invalid');
			$collabAreaFeedback.hide();
		}

		// --- 4. Custom Radio Validation: Specific Project Idea ---
		const $specificProjectRadios = $form.find('input[name="specific_project_idea"]');
		const $specificProjectRadioFeedback = $form.find('.specific-project-feedback');
		if($specificProjectRadios.filter(':checked').length === 0) {
			$specificProjectRadios.addClass('is-invalid');
			$specificProjectRadioFeedback.show();
			formIsValid = false;
		} else {
			$specificProjectRadios.removeClass('is-invalid');
			$specificProjectRadioFeedback.hide();
		}

		// --- 5. Conditional Validation: Specific Project Description ---
		if($specificProjectYesRadio.is(':checked') && $specificProjectDescription.val().trim() === '') {
			$specificProjectDescription.addClass('is-invalid');
			$specificProjectDescriptionFeedback.show();
			formIsValid = false;
		} else {
			$specificProjectDescription.removeClass('is-invalid');
			$specificProjectDescriptionFeedback.hide();
		}

		// --- Follow-Up Preferences: These fields are optional and do not require client-side validation forcing a check. ---
		// Their values will be included in formData if checked.

		// If form is not valid at this point, stop submission and show Bootstrap feedback.
		if(!formIsValid) {
			$form.addClass('was-validated'); // Add Bootstrap's class to show validation feedback.
			$messagesDiv.addClass('alert alert-danger').html(bpi_ajax_object.messages.validation_error);
			return; // Stop AJAX submission.
		}

		// If all validation passes, proceed with AJAX.
		// Disable the submit button and show loading indicator.
		$submitButton.prop('disabled', true).html('<span class="spinner-border spinner-border-sm" role="status" aria-hidden="true"></span> ' + bpi_ajax_object.messages.submitting);

		// Serialize form data.
		const formData = $form.serializeArray(); // Use serializeArray to easily add custom fields.

		// Add nonce field to formData.
		formData.push({name: 'partner_form_nonce_field', value: bpi_ajax_object.nonce});
		formData.push({name: 'action', value: 'bpi_submit_form'}); // Add AJAX action.

		// --- DEBUGGING: Log formData before sending ---
		console.log('BPI JS: Sending formData:', formData);

		// Perform AJAX request.
		$.ajax({
			url: bpi_ajax_object.ajax_url, // WordPress AJAX endpoint.
			type: 'POST',
			data: formData, // Send serialized array.
			dataType: 'json', // Explicitly tell jQuery to expect JSON.
			success: function(response) {
				// --- DEBUGGING: Log raw and parsed server response ---
				//console.log('BPI JS: Raw Server Response (Success Callback):', response);

				if(response.success) {

					form.reset(); // Reset native form fields.
					$form.removeClass('was-validated'); // Remove validation state.
					toggleSpecificProjectDescription(); // Reset conditional fields state
					$messagesDiv.addClass('alert alert-success').html(response.data.message);
					// Scroll to the message div for confirmation
					$('html, body').animate({
						scrollTop: $messagesDiv.offset().top - 50 // -50 for a little padding
					}, 500);
				} else {
					const alertType = response.data.type || 'danger'; // Default to 'danger' if type is not specified.
					$messagesDiv.addClass('alert alert-' + alertType).html(response.data.message);
					$form.addClass('was-validated'); // Keep validation feedback visible on error.
					// Scroll to the message div for error
					$('html, body').animate({
						scrollTop: $messagesDiv.offset().top - 50
					}, 500);
				}
			},
			error: function(jqXHR, textStatus, errorThrown) {
				//console.error('BPI JS: AJAX Error:', textStatus, errorThrown, jqXHR);
				//console.error('BPI JS: Raw Response Text (Error Callback):', jqXHR.responseText);
				$messagesDiv.addClass('alert alert-danger').html(bpi_ajax_object.messages.error);
				$form.addClass('was-validated'); // Keep validation feedback visible on error.
				// Scroll to the message div for error
				$('html, body').animate({
					scrollTop: $messagesDiv.offset().top - 50
				}, 500);
			},
			complete: function() {
				// Re-enable button and restore original text.
				$submitButton.prop('disabled', false).html(originalButtonText);
			}
		});
	});

	// Handle Bootstrap's default validation classes for other inputs on change/blur
	$form.find('input, select, textarea').not('input[type="radio"], input[type="checkbox"]').on('change blur', function() {
		const $input = $(this);
		if($input[0].checkValidity()) {
			$input.removeClass('is-invalid').addClass('is-valid');
			$input.siblings('.invalid-feedback').hide();
		} else {
			$input.addClass('is-invalid').removeClass('is-valid');
			$input.siblings('.invalid-feedback').show();
		}
	});

	// Handle validation for radio/checkbox groups on change
	$form.find('input[type="radio"], input[type="checkbox"]').on('change', function() {
		// For radio groups, apply validation feedback to the group based on any change
		const radioName = $(this).attr('name');
		if(radioName) {
			const $radioGroup = $form.find(`input[name="${radioName}"]`);
			if($radioGroup.filter(':checked').length > 0) {
				$radioGroup.removeClass('is-invalid');
				$radioGroup.closest('.form-check').siblings('.invalid-feedback').hide();
			} else {
				$radioGroup.addClass('is-invalid');
				$radioGroup.closest('.form-check').siblings('.invalid-feedback').show();
			}
		}

		// For checkbox groups, re-evaluate custom validation
		const $orgTypeCheckboxes = $form.find('input[name="type_of_organization[]"]');
		const $orgTypeFeedback = $form.find('.organization-type-feedback');
		if($orgTypeCheckboxes.filter(':checked').length > 0) {
			$orgTypeCheckboxes.removeClass('is-invalid');
			$orgTypeFeedback.hide();
		} else {
			$orgTypeCheckboxes.addClass('is-invalid');
			$orgTypeFeedback.show();
		}

		const $collabAreaCheckboxes = $form.find('input[name="collaboration_areas[]"]');
		const $otherCollabTextarea = $form.find('textarea[name="other_collaboration_areas"]');
		const $collabAreaFeedback = $form.find('.collaboration-areas-feedback');
		if($collabAreaCheckboxes.filter(':checked').length > 0 || $otherCollabTextarea.val().trim() !== '') {
			$collabAreaCheckboxes.removeClass('is-invalid');
			$otherCollabTextarea.removeClass('is-invalid');
			$collabAreaFeedback.hide();
		} else {
			$collabAreaCheckboxes.addClass('is-invalid');
			$otherCollabTextarea.addClass('is-invalid');
			$collabAreaFeedback.show();
		}

		$('#followup_schedule_meeting').removeAttr('required');
		$('#followup_schedule_meeting').removeClass('is-invalid');

		$('#followup_send_packet').removeAttr('required');
		$('#followup_send_packet').removeClass('is-invalid');

		$('#followup_mailing_list').removeAttr('required');
		$('#followup_mailing_list').removeClass('is-invalid');
	});

	// Trigger validation on blur for textareas
	$form.find('textarea').on('blur', function() {
		const $textarea = $(this);
		if($textarea.is('[required]')) { // Only apply if required
			if($textarea.val().trim() !== '') {
				$textarea.removeClass('is-invalid').addClass('is-valid');
				$textarea.siblings('.invalid-feedback').hide();
			} else {
				$textarea.addClass('is-invalid').removeClass('is-valid');
				$textarea.siblings('.invalid-feedback').show();
			}
		}
	});

	// Special handling for the "Other (please describe)" textarea if related checkboxes are unchecked
	$form.find('input[name="collaboration_areas[]"]').on('change', function() {
		const $collabAreaCheckboxes = $form.find('input[name="collaboration_areas[]"]');
		const $otherCollabTextarea = $form.find('textarea[name="other_collaboration_areas"]');
		if($collabAreaCheckboxes.filter(':checked').length === 0) {
			$otherCollabTextarea.attr('required', 'required'); // Make "other" required if no checkboxes
			// Re-validate "other" if it becomes required
			if($otherCollabTextarea.val().trim() === '') {
				$otherCollabTextarea.addClass('is-invalid');
			}
		} else {
			$otherCollabTextarea.removeAttr('required'); // Make "other" optional if checkboxes are checked
			$otherCollabTextarea.removeClass('is-invalid').siblings('.invalid-feedback').hide(); // Clear validation
		}
	});

	// Reset validation states on form reset
	$form[0].addEventListener('reset', function() {
		$form.removeClass('was-validated');
		$form.find('.is-valid, .is-invalid').removeClass('is-valid is-invalid');
		$form.find('.invalid-feedback').hide();
		$messagesDiv.empty().removeClass('alert alert-success alert-warning alert-danger');
		toggleSpecificProjectDescription(); // Reset conditional visibility
	});
});