<?php
/**
 * Plugin Name: Bootstrap Advance Contact Form
 * Plugin URI: https://www.emad-zedan.com
 * Description: This plugin provides a Bootstrap-styled contact form with AJAX submission. It saves all submitted data to a custom post type and provides a dedicated WordPress admin settings page under 'Contact Messages' to set a dynamic admin notification email, compatible with the free ACF plugin.
 * Version: 1.0.0
 * Requires at least: 5.2
 * Requires PHP: 8.0
 * Author: Emad Zedan
 * Author URI: https://www.emad-zedan.com
 * License: GPLv2 or Later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.txt
 * Text Domain: bootstrap-advance-contact-form
 * Domain Path: /languages
 */

if ( !defined( 'ABSPATH' ) ) {
    exit; // Exit if accessed directly.
}

// Define plugin constants for easier path management.
define( 'BACF_PLUGIN_DIR', plugin_dir_path( __FILE__ ) );
define( 'BACF_PLUGIN_URL', plugin_dir_url( __FILE__ ) );

// Admin CPT Registration
add_action('init', 'bacf_create_contact_cpt');

// Admin CSS & JS Enqueue
add_action('admin_enqueue_scripts', 'bacf_admin_css');
add_action('admin_enqueue_scripts', 'bacf_admin_js');

// Front-end CSS & JS Enqueue
add_action('wp_enqueue_scripts', 'bacf_wp_enqueue_assets');

// Front-end Shortcode
add_shortcode('bootstrap-advance-contact-form-wp-shortcode', 'bacf_wp_shortcode');

// AJAX Handler for form submission
add_action('wp_ajax_bacf_submit_form', 'bacf_handle_ajax_form_submission');
add_action('wp_ajax_nopriv_bacf_submit_form', 'bacf_handle_ajax_form_submission');

// Admin CPT: Add custom columns to list table
add_filter('manage_bs_contact_cpt_posts_columns', 'bacf_add_contact_columns');
add_action('manage_bs_contact_cpt_posts_custom_column', 'bacf_populate_contact_columns', 10, 2);

// Register WordPress settings page for plugin settings (replaces ACF Options Page)
add_action('admin_menu', 'bacf_add_settings_page');
add_action('admin_init', 'bacf_register_settings');

// Add dashboard widget
add_action('wp_dashboard_setup', 'bacf_add_dashboard_widgets');

// Removed screen option filters as per request
// remove_filter('screen_settings', 'bacf_contact_messages_screen_options', 10, 2);
// remove_filter('set-screen-option', 'bacf_set_contact_messages_screen_option', 10, 3);


/**
 * Register Custom Post Type: Contact Messages (bs_contact_cpt)
 * This CPT will store all form submissions.
 */
function bacf_create_contact_cpt(){
    $labels = array(
        'name'                  => _x( 'Contact Messages', 'Post Type General Name', 'bootstrap-advance-contact-form' ),
        'singular_name'         => _x( 'Contact Message', 'Post Type Singular Name', 'bootstrap-advance-contact-form' ),
        'menu_name'             => __( 'Contact Messages', 'bootstrap-advance-contact-form' ),
        'name_admin_bar'        => __( 'Contact Message', 'bootstrap-advance-contact-form' ),
        'archives'              => __( 'Message Archives', 'bootstrap-advance-contact-form' ),
        'attributes'            => __( 'Message Attributes', 'bootstrap-advance-contact-form' ),
        'parent_item_colon'     => __( 'Parent Message:', 'bootstrap-advance-contact-form' ),
        'all_items'             => __( 'All Messages', 'bootstrap-advance-contact-form' ),
        'add_new_item'          => __( 'Add New Message', 'bootstrap-advance-contact-form' ),
        'add_new'               => __( 'Add New', 'bootstrap-advance-contact-form' ),
        'new_item'              => __( 'New Message', 'bootstrap-advance-contact-form' ),
        'edit_item'             => __( 'Edit Message', 'bootstrap-advance-contact-form' ),
        'update_item'           => __( 'Update Message', 'bootstrap-advance-contact-form' ),
        'view_item'             => __( 'View Message', 'bootstrap-advance-contact-form' ),
        'view_items'            => __( 'View Messages', 'bootstrap-advance-contact-form' ),
        'search_items'          => __( 'Search Messages', 'bootstrap-advance-contact-form' ),
        'not_found'             => __( 'Not found', 'bootstrap-advance-contact-form' ),
        'not_found_in_trash'    => __( 'Not found in Trash', 'bootstrap-advance-contact-form' ),
        'featured_image'        => __( 'Featured Image', 'bootstrap-advance-contact-form' ),
        'set_featured_image'    => __( 'Set featured image', 'bootstrap-advance-contact-form' ),
        'remove_featured_image' => __( 'Remove featured image', 'bootstrap-advance-contact-form' ),
        'use_featured_image'    => __( 'Use as featured image', 'bootstrap-advance-contact-form' ),
        'insert_into_item'      => __( 'Insert into message', 'bootstrap-advance-contact-form' ),
        'uploaded_to_this_item' => __( 'Uploaded to this message', 'bootstrap-advance-contact-form' ),
        'items_list'            => __( 'Messages list', 'bootstrap-advance-contact-form' ),
        'items_list_navigation' => __( 'Messages list navigation', 'bootstrap-advance-contact-form' ),
        'filter_items_list'     => __( 'Filter messages list', 'bootstrap-advance-contact-form' ),
    );

    $args = array(
        'label'                 => __( 'Contact Message', 'bootstrap-advance-contact-form' ),
        'description'           => __( 'Contact form submissions from the website.', 'bootstrap-advance-contact-form' ),
        'labels'                => $labels,
        'supports'              => array( 'title' ), // Removed 'editor' to prevent saving to post_content by default
        'hierarchical'          => false,
        'public'                => false, // Not publicly viewable on front-end
        'show_ui'               => true,
        'show_in_menu'          => true,
        'menu_position'         => 25, // Position in admin menu (below comments)
        'menu_icon'             => 'dashicons-email-alt', // Email icon
        'show_in_admin_bar'     => true,
        'show_in_nav_menus'     => false, // Not for navigation menus
        'can_export'            => true,
        'has_archive'           => false,
        'exclude_from_search'   => true,
        'publicly_queryable'    => false,
        'capability_type'       => 'post',
        'rewrite'               => false,
    );

    register_post_type('bs_contact_cpt', $args);
}


/**
 * Admin CSS & JS Enqueue
 * Enqueues styles and scripts for the WordPress admin area.
 */
function bacf_admin_css(){
       //wp_enqueue_style('bacf-admin-css', BACF_PLUGIN_URL . "assets/admin/style/css/admin-css.css", array(), '1.0.0', 'all');
}

function bacf_admin_js(){
    //wp_enqueue_script('jquery');
    //wp_enqueue_script('bacf-admin-js', BACF_PLUGIN_URL . "assets/admin/js/admin-js.js", array(), '1.0.0', true);
}


/**
 * Front-end CSS & JS Enqueue
 * Enqueues styles and scripts for the front-end of the website.
 */
function bacf_wp_enqueue_assets(){
    // intl-tel-input CSS
    wp_enqueue_style('bacf-intl-tel-input-css', "https://cdnjs.cloudflare.com/ajax/libs/intl-tel-input/17.0.8/css/intlTelInput.css", array(), '17.0.8', 'all');
    // Plugin WP Custom Style
    wp_enqueue_style('bacf-wp-css', BACF_PLUGIN_URL . "assets/wp/style/css/wp-css.min.css", array('bacf-intl-tel-input-css'), '1.0.2', 'all' );
    wp_enqueue_style('bacf-contact-css', BACF_PLUGIN_URL . "assets/wp/style/css/contact.min.css", array('bacf-intl-tel-input-css', 'bacf-wp-css'), '1.0.3', 'all' );

    // jQuery
    wp_enqueue_script('bacf-jQuery', "https://code.jquery.com/jquery-3.7.1.min.js", array(), '3.7.1', true);
    // Explicitly enqueue utils.js as a dependency for intlTelInput
    wp_enqueue_script('bacf-intl-tel-input-utils-js', "https://cdnjs.cloudflare.com/ajax/libs/intl-tel-input/17.0.8/js/utils.js", array('bacf-jQuery'), '17.0.8', true);
    // intl-tel-input JS, now depends on utils.js
    wp_enqueue_script('bacf-intl-tel-input-js', "https://cdnjs.cloudflare.com/ajax/libs/intl-tel-input/17.0.8/js/intlTelInput.min.js", array('bacf-jQuery', 'bacf-intl-tel-input-utils-js'), '17.0.8', true);
    // Plugin WP JS (for AJAX form handling), now depends on intl-tel-input-js
    wp_enqueue_script('bacf-wp-js', BACF_PLUGIN_URL . "assets/wp/js/wp-js.js", array('bacf-jQuery', 'bacf-intl-tel-input-js'), '1.0.0', true);

    // Localize script to pass AJAX URL and nonce to JavaScript
    wp_localize_script(
        'bacf-wp-js', // Handle of the script to localize
        'bacf_ajax_object', // Name of the JS object
        array(
            'ajax_url' => admin_url( 'admin-ajax.php' ),
            'nonce'    => wp_create_nonce( 'bacf_form_nonce' ), // Unique nonce for this form
            'messages' => array(
                'success'          => __( 'Thank you! Your message has been sent successfully.', 'bootstrap-advance-contact-form' ),
                'email_error'      => __( 'Your message was saved, but there was an issue sending the confirmation email. Please contact support.', 'bootstrap-advance-contact-form' ),
                'error'            => __( 'There was an error sending your message. Please ensure all fields are correctly filled and try again.', 'bootstrap-advance-contact-form' ),
                'validation_error' => __( 'Please fill in all required fields correctly.', 'bootstrap-advance-contact-form' ),
                'security_error'   => __( 'Security check failed. Please try again.', 'bootstrap-advance-contact-form' ),
                'submitting'       => __( 'Sending...', 'bootstrap-advance-contact-form' ), // Added for button loading text
            ),
        )
    );
}

/**
 * Shortcode to display the contact form.
 * Uses Bootstrap classes for styling.
 */
function bacf_wp_shortcode(){
    ob_start(); // Start output buffering
    ?>
    <div class="contactForm px-0">
        <div class="row">
            <div class="col-12 col-sm-12 col-md-12 col-lg-6 col-xl-5 col-xxl-5">
                <img class="w-100" src="<?php echo BACF_PLUGIN_URL . 'assets/wp/img/contact.jpg'; ?>" alt="<?php esc_attr_e('Contact Image', 'bootstrap-advance-contact-form'); ?>">
            </div>
            <div class="col-12 col-sm-12 col-md-12 col-lg-6 col-xl-7 col-xxl-7">
                <form method="post" class="bootstrap-advanced-contact-form__form needs-validation p-1" novalidate>
                    <h2 class="contactGetInTouch text-center text-sm-center text-md-center text-lg-start text-xl-start text-xxl-start"><?php _e('Please Fill The Following Form:', 'bootstrap-advance-contact-form'); ?></h2>
                    <hr>
                    <div class="row mt-2">
                        <div class="col-12 col-sm-12 col-md-6 col-lg-6 col-xl-6 col-xxl-6 mt-2">
                            <input class="form-control" type="text" name="firstname" placeholder="<?php _e('First Name', 'bootstrap-advance-contact-form'); ?>" required>
                            <div class="invalid-feedback"><?php _e('Please enter your first name.', 'bootstrap-advance-contact-form'); ?></div>
                        </div>
                        <div class="col-12 col-sm-12 col-md-6 col-lg-6 col-xl-6 col-xxl-6 mt-2">
                            <input class="form-control" type="text" name="lastname" placeholder="<?php _e('Last Name', 'bootstrap-advance-contact-form'); ?>" required>
                            <div class="invalid-feedback"><?php _e('Please enter your last name.', 'bootstrap-advance-contact-form'); ?></div>
                        </div>
                    </div>

                    <div class="row">
                        <div class="col-12 col-sm-12 col-md-6 col-lg-6 col-xl-6 col-xxl-6 mt-2">
                            <input class="form-control internationalphone w-100" type="tel" placeholder="<?php _e('International Phone', 'bootstrap-advance-contact-form'); ?>" name="myinternationalphone" required value="+1">
                            <div class="invalid-feedback"><?php _e('Please enter a valid phone number.', 'bootstrap-advance-contact-form'); ?></div>
                        </div>
                        <div class="col-12 col-sm-12 col-md-6 col-lg-6 col-xl-6 col-xxl-6 mt-2">
                            <input class="form-control" type="email" name="email" placeholder="<?php _e('Email', 'bootstrap-advance-contact-form'); ?>" required>
                            <div class="invalid-feedback"><?php _e('Please enter a valid email address.', 'bootstrap-advance-contact-form'); ?></div>
                        </div>
                    </div>

                    <select class="form-select mt-2" name="country" required>
                        <option value="" selected disabled><?php _e('Country', 'bootstrap-advance-contact-form'); ?></option>
                        <option value="Afganistan">Afghanistan</option>
                        <option value="Albania">Albania</option>
                        <option value="Algeria">Algeria</option>
                        <option value="American Samoa">American Samoa</option>
                        <option value="Andorra">Andorra</option>
                        <option value="Angola">Angola</option>
                        <option value="Anguilla">Anguilla</option>
                        <option value="Antigua & Barbuda">Antigua & Barbuda</option>
                        <option value="Argentina">Argentina</option>
                        <option value="Armenia">Armenia</option>
                        <option value="Aruba">Aruba</option>
                        <option value="Australia">Australia</option>
                        <option value="Austria">Austria</option>
                        <option value="Azerbaijan">Azerbaijan</option>
                        <option value="Bahamas">Bahamas</option>
                        <option value="Bahrain">Bahrain</option>
                        <option value="Bangladesh">Bangladesh</option>
                        <option value="Barbados">Barbados</option>
                        <option value="Belarus">Belarus</option>
                        <option value="Belgium">Belgium</option>
                        <option value="Belize">Belize</option>
                        <option value="Benin">Benin</option>
                        <option value="Bermuda">Bermuda</option>
                        <option value="Bhutan">Bhutan</option>
                        <option value="Bolivia">Bolivia</option>
                        <option value="Bonaire">Bonaire</option>
                        <option value="Bosnia & Herzegovina">Bosnia & Herzegovina</option>
                        <option value="Botswana">Botswana</option>
                        <option value="Brazil">Brazil</option>
                        <option value="British Indian Ocean Ter">British Indian Ocean Ter</option>
                        <option value="Brunei">Brunei</option>
                        <option value="Bulgaria">Bulgaria</option>
                        <option value="Burkina Faso">Burkina Faso</option>
                        <option value="Burundi">Burundi</option>
                        <option value="Cambodia">Cambodia</option>
                        <option value="Cameroon">Cameroon</option>
                        <option value="Canada">Canada</option>
                        <option value="Canary Islands">Canary Islands</option>
                        <option value="Cape Verde">Cape Verde</option>
                        <option value="Cayman Islands">Cayman Islands</option>
                        <option value="Central African Republic">Central African Republic</option>
                        <option value="Chad">Chad</option>
                        <option value="Channel Islands">Channel Islands</option>
                        <option value="Chile">Chile</option>
                        <option value="China">China</option>
                        <option value="Christmas Island">Christmas Island</option>
                        <option value="Cocos Island">Cocos Island</option>
                        <option value="Colombia">Colombia</option>
                        <option value="Comoros">Comoros</option>
                        <option value="Congo">Congo</option>
                        <option value="Cook Islands">Cook Islands</option>
                        <option value="Costa Rica">Costa Rica</option>
                        <option value="Cote DIvoire">Cote DIvoire</option>
                        <option value="Croatia">Croatia</option>
                        <option value="Cuba">Cuba</option>
                        <option value="Curaco">Curacao</option>
                        <option value="Cyprus">Cyprus</option>
                        <option value="Czech Republic">Czech Republic</option>
                        <option value="Denmark">Denmark</option>
                        <option value="Djibouti">Djibouti</option>
                        <option value="Dominica">Dominica</option>
                        <option value="Dominican Republic">Dominican Republic</option>
                        <option value="East Timor">East Timor</option>
                        <option value="Ecuador">Ecuador</option>
                        <option value="Egypt">Egypt</option>
                        <option value="El Salvador">El Salvador</option>
                        <option value="Equatorial Guinea">Equatorial Guinea</option>
                        <option value="Eritrea">Eritrea</option>
                        <option value="Estonia">Estonia</option>
                        <option value="Ethiopia">Ethiopia</option>
                        <option value="Falkland Islands">Falkland Islands</option>
                        <option value="Faroe Islands">Faroe Islands</option>
                        <option value="Fiji">Fiji</option>
                        <option value="Finland">Finland</option>
                        <option value="France">France</option>
                        <option value="French Guiana">French Guiana</option>
                        <option value="French Polynesia">French Polynesia</option>
                        <option value="French Southern Ter">French Southern Ter</option>
                        <option value="Gabon">Gabon</option>
                        <option value="Gambia">Gambia</option>
                        <option value="Georgia">Georgia</option>
                        <option value="Germany">Germany</option>
                        <option value="Ghana">Ghana</option>
                        <option value="Gibraltar">Gibraltar</option>
                        <option value="Great Britain">Great Britain</option>
                        <option value="Greece">Greece</option>
                        <option value="Greenland">Greenland</option>
                        <option value="Grenada">Grenada</option>
                        <option value="Guadeloupe">Guadeloupe</option>
                        <option value="Guam">Guam</option>
                        <option value="Guatemala">Guatemala</option>
                        <option value="Guinea">Guinea</option>
                        <option value="Guyana">Guyana</option>
                        <option value="Haiti">Haiti</option>
                        <option value="Hawaii">Hawaii</option>
                        <option value="Honduras">Honduras</option>
                        <option value="Hong Kong">Hong Kong</option>
                        <option value="Hungary">Hungary</option>
                        <option value="Iceland">Iceland</option>
                        <option value="Indonesia">Indonesia</option>
                        <option value="India">India</option>
                        <option value="Iran">Iran</option>
                        <option value="Iraq">Iraq</option>
                        <option value="Ireland">Ireland</option>
                        <option value="Isle of Man">Isle of Man</option>
                        <option value="Israel">Israel</option>
                        <option value="Italy">Italy</option>
                        <option value="Jamaica">Jamaica</option>
                        <option value="Japan">Japan</option>
                        <option value="Jordan">Jordan</option>
                        <option value="Kazakhstan">Kazakhstan</option>
                        <option value="Kenya">Kenya</option>
                        <option value="Kiribati">Kiribati</option>
                        <option value="Korea North">Korea North</option>
                        <option value="Korea Sout">Korea South</option>
                        <option value="Kuwait">Kuwait</option>
                        <option value="Kyrgyzstan">Kyrgyzstan</option>
                        <option value="Laos">Laos</option>
                        <option value="Latvia">Latvia</option>
                        <option value="Lebanon">Lebanon</option>
                        <option value="Lesotho">Lesotho</option>
                        <option value="Liberia">Liberia</option>
                        <option value="Libya">Libya</option>
                        <option value="Liechtenstein">Liechtenstein</option>
                        <option value="Lithuania">Lithuania</option>
                        <option value="Luxembourg">Luxembourg</option>
                        <option value="Macau">Macau</option>
                        <option value="Macedonia">Macedonia</option>
                        <option value="Madagascar">Madagascar</option>
                        <option value="Malaysia">Malaysia</option>
                        <option value="Malawi">Malawi</option>
                        <option value="Maldives">Maldives</option>
                        <option value="Mali">Mali</option>
                        <option value="Malta">Malta</option>
                        <option value="Marshall Islands">Marshall Islands</option>
                        <option value="Martinique">Martinique</option>
                        <option value="Mauritania">Mauritania</option>
                        <option value="Mauritius">Mauritius</option>
                        <option value="Mayotte">Mayotte</option>
                        <option value="Mexico">Mexico</option>
                        <option value="Midway Islands">Midway Islands</option>
                        <option value="Moldova">Moldova</option>
                        <option value="Monaco">Monaco</option>
                        <option value="Mongolia">Mongolia</option>
                        <option value="Montserrat">Montserrat</option>
                        <option value="Morocco">Morocco</option>
                        <option value="Mozambique">Mozambique</option>
                        <option value="Myanmar">Myanmar</option>
                        <option value="Nambia">Nambia</option>
                        <option value="Nauru">Nauru</option>
                        <option value="Nepal">Nepal</option>
                        <option value="Netherland Antilles">Netherland Antilles</option>
                        <option value="Netherlands">Netherlands (Holland, Europe)</option>
                        <option value="Nevis">Nevis</option>
                        <option value="New Caledonia">New Caledonia</option>
                        <option value="New Zealand">New Zealand</option>
                        <option value="Nicaragua">Nicaragua</option>
                        <option value="Niger">Niger</option>
                        <option value="Nigeria">Nigeria</option>
                        <option value="Niue">Niue</option>
                        <option value="Norfolk Island">Norfolk Island</option>
                        <option value="Norway">Norway</option>
                        <option value="Oman">Oman</option>
                        <option value="Pakistan">Pakistan</option>
                        <option value="Palau Island">Palau Island</option>
                        <option value="Palestine">Palestine</option>
                        <option value="Panama">Panama</option>
                        <option value="Papua New Guinea">Papua New Guinea</option>
                        <option value="Paraguay">Paraguay</option>
                        <option value="Peru">Peru</option>
                        <option value="Phillipines">Philippines</option>
                        <option value="Pitcairn Island">Pitcairn Island</option>
                        <option value="Poland">Poland</option>
                        <option value="Portugal">Portugal</option>
                        <option value="Puerto Rico">Puerto Rico</option>
                        <option value="Qatar">Qatar</option>
                        <option value="Republic of Montenegro">Republic of Montenegro</option>
                        <option value="Republic of Serbia">Republic of Serbia</option>
                        <option value="Reunion">Reunion</option>
                        <option value="Romania">Romania</option>
                        <option value="Russia">Russia</option>
                        <option value="Rwanda">Rwanda</option>
                        <option value="St Barthelemy">St Barthelemy</option>
                        <option value="St Eustatius">St Eustatius</option>
                        <option value="St Helena">St Helena</option>
                        <option value="St Kitts-Nevis">St Kitts-Nevis</option>
                        <option value="St Lucia">St Lucia</option>
                        <option value="St Maarten">St Maarten</option>
                        <option value="St Pierre & Miquelon">St Pierre & Miquelon</option>
                        <option value="St Vincent & Grenadines">St Vincent & Grenadines</option>
                        <option value="Saipan">Saipan</option>
                        <option value="Samoa">Samoa</option>
                        <option value="Samoa American">Samoa American</option>
                        <option value="San Marino">San Marino</option>
                        <option value="Sao Tome & Principe">Sao Tome & Principe</option>
                        <option value="Saudi Arabia">Saudi Arabia</option>
                        <option value="Senegal">Senegal</option>
                        <option value="Seychelles">Seychelles</option>
                        <option value="Sierra Leone">Sierra Leone</option>
                        <option value="Singapore">Singapore</option>
                        <option value="Slovakia">Slovakia</option>
                        <option value="Slovenia">Slovenia</option>
                        <option value="Solomon Islands">Solomon Islands</option>
                        <option value="Somalia">Somalia</option>
                        <option value="South Africa">South Africa</option>
                        <option value="Spain">Spain</option>
                        <option value="Sri Lanka">Sri Lanka</option>
                        <option value="Sudan">Sudan</option>
                        <option value="Suriname">Suriname</option>
                        <option value="Swaziland">Swaziland</option>
                        <option value="Sweden">Sweden</option>
                        <option value="Switzerland">Switzerland</option>
                        <option value="Syria">Syria</option>
                        <option value="Tahiti">Tahiti</option>
                        <option value="Taiwan">Taiwan</option>
                        <option value="Tajikistan">Tajikistan</option>
                        <option value="Tanzania">Tanzania</option>
                        <option value="Thailand">Thailand</option>
                        <option value="Togo">Togo</option>
                        <option value="Tokelau">Tokelau</option>
                        <option value="Tonga">Tonga</option>
                        <option value="Trinidad & Tobago">Trinidad & Tobago</option>
                        <option value="Tunisia">Tunisia</option>
                        <option value="Turkey">Turkey</option>
                        <option value="Turkmenistan">Turkmenistan</option>
                        <option value="Turks & Caicos Is">Turks & Caicos Is</option>
                        <option value="Tuvalu">Tuvalu</option>
                        <option value="Uganda">Uganda</option>
                        <option value="United Kingdom">United Kingdom</option>
                        <option value="Ukraine">Ukraine</option>
                        <option value="United Arab Erimates">United Arab Emirates</option>
                        <option value="United States of America">United States of America</option>
                        <option value="Uraguay">Uruguay</option>
                        <option value="Uzbekistan">Uzbekistan</option>
                        <option value="Vanuatu">Vanuatu</option>
                        <option value="Vatican City State">Vatican City State</option>
                        <option value="Venezuela">Venezuela</option>
                        <option value="Vietnam">Vietnam</option>
                        <option value="Virgin Islands (Brit)">Virgin Islands (Brit)</option>
                        <option value="Virgin Islands (USA)">Virgin Islands (USA)</option>
                        <option value="Wake Island">Wake Island</option>
                        <option value="Wallis & Futana Is">Wallis & Futana Is</option>
                        <option value="Yemen">Yemen</option>
                        <option value="Zaire">Zaire</option>
                        <option value="Zambia">Zambia</option>
                        <option value="Zimbabwe">Zimbabwe</option>
                    </select>
                    <div class="invalid-feedback"><?php _e('Please select your country.', 'bootstrap-advance-contact-form'); ?></div>

                    <div class="row">
                        <div class="col-12 col-sm-12 col-md-6 col-lg-6 col-xl-6 col-xxl-6 mt-2">
                            <input class="form-control" type="text" name="company" placeholder="<?php _e('Company', 'bootstrap-advance-contact-form'); ?>" required>
                            <div class="invalid-feedback"><?php _e('Please enter your company name.', 'bootstrap-advance-contact-form'); ?></div>
                        </div>
                        <div class="col-12 col-sm-12 col-md-6 col-lg-6 col-xl-6 col-xxl-6 mt-2">
                            <input class="form-control" type="text" name="position" placeholder="<?php _e('Position', 'bootstrap-advance-contact-form'); ?>" required>
                            <div class="invalid-feedback"><?php _e('Please enter your position.', 'bootstrap-advance-contact-form'); ?></div>
                        </div>
                    </div>

                    <!-- Subject field changed from select to text input -->
                    <input class="form-control mt-2" type="text" name="subject" placeholder="<?php _e('Subject', 'bootstrap-advance-contact-form'); ?>" required>
                    <div class="invalid-feedback"><?php _e('Please enter a subject.', 'bootstrap-advance-contact-form'); ?></div>

                    <textarea class="form-control mt-2" name="message" required placeholder="<?php _e('Message', 'bootstrap-advance-contact-form'); ?>" rows="10"></textarea>
                    <div class="invalid-feedback"><?php _e('Please enter your message.', 'bootstrap-advance-contact-form'); ?></div>

                    <hr class="pt-2 pb-0 mb-0">
                    <!-- Removed wp_nonce_field from here to prevent caching issues with nonces. -->
                    <!-- The nonce is now exclusively passed via wp_localize_script to JavaScript. -->
                    <div class="row">
                        <div class="col-12 col-sm-12 col-md-12 col-lg-9 col-xl-9 col-xxl-9 contactPrivacy mt-2 text-center text-sm-center text-md-center text-lg-start text-xl-start text-xxl-start" style="font-size: 0.8rem !important;">
                            <?php _e('By clicking submit, you agree to the process of personal data according to the privacy policy, This form is strictly for business partnership related to queries only.', 'bootstrap-advance-contact-form'); ?>
                        </div>
                        <div class="col-12 col-sm-12 col-md-12 col-lg-3 col-xl-3 col-xxl-3 mt-2 d-flex justify-content-center justify-content-sm-center justify-content-md-center justify-content-lg-end justify-content-xl-end justify-content-xxl-end">
                            <button class="btn btn-primary px-3" style="white-space: nowrap;" type="submit" name="submit" value="submit"><?php _e('Send Message', 'bootstrap-advance-contact-form'); ?></button>
                        </div>
                    </div>
                    <div class="bacf-form-messages mt-3">
                        <!-- AJAX messages will be displayed here -->
                    </div>
                </form>
            </div>
        </div>
    </div>
    <?php
    return ob_get_clean(); // Return the buffered content
}


/**
 * Handle AJAX form submission for the contact form.
 * IMPORTANT: This function now relies exclusively on Advanced Custom Fields (ACF)
 * to save custom data. Ensure ACF is installed, active, and the corresponding
 * fields are created for the 'Contact Messages' (bs_contact_cpt) post type.
 */
function bacf_handle_ajax_form_submission() {
    // --- DEBUGGING: Log raw $_POST data ---
    error_log('BACF AJAX: Received $_POST data: ' . print_r($_POST, true));

    // Verify nonce for security.
    // Corrected nonce action name to match wp_create_nonce()
    if ( ! isset( $_POST['bacf_form_nonce_field'] ) || ! wp_verify_nonce( $_POST['bacf_form_nonce_field'], 'bacf_form_nonce' ) ) {
        error_log('BACF AJAX: Nonce verification failed.');
        wp_send_json_error( array( 'message' => __( 'Security check failed. Please try again.', 'bootstrap-advance-contact-form' ) ) );
        wp_die();
    }

    // Sanitize and validate inputs.
    $firstname = sanitize_text_field( $_POST['firstname'] );
    $lastname  = sanitize_text_field( $_POST['lastname'] );
    $phone     = sanitize_text_field( $_POST['myinternationalphone'] ); // Assuming this is the field name from intl-tel-input
    $email     = sanitize_email( $_POST['email'] );
    $country   = sanitize_text_field( $_POST['country'] );
    $company   = sanitize_text_field( $_POST['company'] );
    $position  = sanitize_text_field( $_POST['position'] );
    $subject   = sanitize_text_field( $_POST['subject'] ); // Subject is now a text field
    $message   = sanitize_textarea_field( $_POST['message'] );

    // Basic server-side validation.
    if ( empty( $firstname ) || empty( $lastname ) || empty( $phone ) || empty( $email ) || ! is_email( $email ) || empty( $country ) || empty( $company ) || empty( $position ) || empty( $subject ) || empty( $message ) ) {
        error_log('BACF AJAX: Validation failed for required fields.');
        wp_send_json_error( array( 'message' => __( 'Please fill in all required fields correctly.', 'bootstrap-advance-contact-form' ) ) ) ;
        wp_die();
    }

    // Prepare post data for the new custom post type entry.
    $post_data = array(
        'post_title'    => sprintf( __( 'Message from %s %s', 'bootstrap-advance-contact-form' ), $firstname, $lastname ),
        // Removed 'post_content' => $message to prevent saving message twice.
        // Message will now ONLY be saved in the 'contact_message' ACF field.
        'post_status'   => 'publish',
        'post_type'     => 'bs_contact_cpt',
    );

    // Insert the new post into the WordPress database.
    $post_id = wp_insert_post( $post_data );

    // --- DEBUGGING: Log wp_insert_post result ---
    error_log('BACF AJAX: wp_insert_post result for post_id: ' . print_r($post_id, true));
    if ( is_wp_error( $post_id ) ) {
        error_log('BACF AJAX: wp_insert_post error: ' . $post_id->get_error_message());
    }


    if ( ! is_wp_error( $post_id ) ) {
        // Save custom fields using Advanced Custom Fields (ACF).
        if ( function_exists( 'update_field' ) ) {
            // --- DEBUGGING: Log ACF update_field calls and results ---
            $update_results = [];
            $update_results['firstname'] = update_field( 'contact_firstname', $firstname, $post_id );
            $update_results['lastname'] = update_field( 'contact_lastname', $lastname, $post_id );
            $update_results['phone'] = update_field( 'contact_phone', $phone, $post_id );
            $update_results['email'] = update_field( 'contact_email', $email, $post_id );
            $update_results['country'] = update_field( 'contact_country', $country, $post_id );
            $update_results['company'] = update_field( 'contact_company', $company, $post_id );
            $update_results['position'] = update_field( 'contact_position', $position, $post_id );
            $update_results['subject'] = update_field( 'contact_subject', $subject, $post_id );
            // This line now exclusively saves the message content to the ACF field
            $update_results['message'] = update_field( 'contact_message', $message, $post_id );
            error_log('BACF AJAX: ACF update_field results for post ID ' . $post_id . ': ' . print_r($update_results, true));

        } else {
            // Log an error if ACF is not active and custom fields cannot be saved.
            error_log( 'ACF is NOT active. Contact form custom fields were NOT saved for post ID: ' . $post_id . '. Please install and activate Advanced Custom Fields plugin.' );
            // For this version, we are explicitly relying on ACF.
        }

        // Retrieve admin email from WordPress options (set via new settings page)
        $admin_email_recipient = get_option( 'bacf_admin_email_recipient', get_option( 'admin_email' ) ); // Fallback to WP admin email

        $site_name   = get_bloginfo( 'name' );

        // Email to Admin
        $to_admin = $admin_email_recipient; // Use the dynamic admin email
        $subject_admin = sprintf( __( 'New Contact Form Submission: %s', 'bootstrap-advance-contact-form' ), $subject );
        $message_admin = '<p>' . __( 'You have received a new contact form submission:', 'bootstrap-advance-contact-form' ) . '</p>';
        $message_admin .= '<ul>';
        $message_admin .= '<li><strong>' . __( 'Name:', 'bootstrap-advance-contact-form' ) . '</strong> ' . esc_html( $firstname . ' ' . $lastname ) . '</li>';
        $message_admin .= '<li><strong>' . __( 'Email:', 'bootstrap-advance-contact-form' ) . '</strong> ' . esc_html( $email ) . '</li>';
        $message_admin .= '<li><strong>' . __( 'Phone:', 'bootstrap-advance-contact-form' ) . '</strong> ' . esc_html( $phone ) . '</li>';
        $message_admin .= '<li><strong>' . __( 'Country:', 'bootstrap-advance-contact-form' ) . '</strong> ' . esc_html( $country ) . '</li>';
        $message_admin .= '<li><strong>' . __( 'Company:', 'bootstrap-advance-contact-form' ) . '</strong> ' . esc_html( $company ) . '</li>';
        $message_admin .= '<li><strong>' . __( 'Position:', 'bootstrap-advance-contact-form' ) . '</strong> ' . esc_html( $position ) . '</li>';
        $message_admin .= '<li><strong>' . __( 'Subject:', 'bootstrap-advance-contact-form' ) . '</strong> ' . esc_html( $subject ) . '</li>';
        $message_admin .= '<li><strong>' . __( 'Message:', 'bootstrap-advance-contact-form' ) . '</strong><br>' . wp_kses_post( nl2br( $message ) ) . '</li>'; // nl2br to preserve line breaks
        $message_admin .= '</ul>';
        $message_admin .= '<p>' . sprintf( __( 'View this message in WordPress admin: %s', 'bootstrap-advance-contact-form' ), get_edit_post_link( $post_id ) ) . '</p>';
        $headers_admin = array( 'Content-Type: text/html; charset=UTF-8', 'From: ' . $site_name . ' <' . $admin_email_recipient . '>' ); // Use the dynamic admin email for 'From'
        $mail_sent_admin = wp_mail( $to_admin, $subject_admin, $message_admin, $headers_admin );
        error_log('BACF AJAX: Admin email sent status: ' . ($mail_sent_admin ? 'Success' : 'Failed'));


        // Email to User (Confirmation)
        $to_user = $email;
        $subject_user = sprintf( __( 'Thank you for your message to %s', 'bootstrap-advance-contact-form' ), $site_name );
        $message_user = '<p>' . sprintf( __( 'Hello %s,', 'bootstrap-advance-contact-form' ), esc_html( $firstname ) ) . '</p>';
        $message_user .= '<p>' . __( 'Thank you for contacting us. We have received your message and will get back to you shortly.', 'bootstrap-advance-contact-form' ) . '</p>';
        $message_user .= '<p>' . __( 'Here is a summary of your submission:', 'bootstrap-advance-contact-form' ) . '</p>';
        $message_user .= '<ul>';
        $message_user .= '<li><strong>' . __( 'Subject:', 'bootstrap-advance-contact-form' ) . '</strong> ' . esc_html( $subject ) . '</li>';
        $message_user .= '<li><strong>' . __( 'Your Message:', 'bootstrap-advance-contact-form' ) . '</strong><br>' . wp_kses_post( nl2br( $message ) ) . '</li>';
        $message_user .= '</ul>';
        $message_user .= '<p>' . sprintf( __( 'Best regards,<br>%s Team', 'bootstrap-advance-contact-form' ), $site_name ) . '</p>';
        $headers_user = array( 'Content-Type: text/html; charset=UTF-8', 'From: ' . $site_name . ' <' . $admin_email_recipient . '>' ); // Use the dynamic admin email for 'From'
        $mail_sent_user = wp_mail( $to_user, $subject_user, $message_user, $headers_user );
        error_log('BACF AJAX: User confirmation email sent status: ' . ($mail_sent_user ? 'Success' : 'Failed'));


        if ( ! $mail_sent_user ) {
            error_log( 'BACF Plugin: Failed to send confirmation email to ' . $email . ' for post ID ' . $post_id );
            // Directly use the translated string for the message.
            wp_send_json_success( array( 'message' => __( 'Your message was saved, but there was an issue sending the confirmation email. Please contact support.', 'bootstrap-advance-contact-form' ), 'type' => 'warning' ) );
        } else {
            // Directly use the translated string for the message.
            wp_send_json_success( array( 'message' => __( 'Thank you! Your message has been sent successfully.', 'bootstrap-advance-contact-form' ), 'type' => 'success' ) );
        }

    } else {
        wp_send_json_error( array( 'message' => __( 'There was an error sending your message. Please ensure all fields are correctly filled and try again.', 'bootstrap-advance-contact-form' ) ) );
    }

    wp_die(); // Always die in AJAX handlers.
}


/**
 * Add custom columns to the 'Contact Messages' list table in the WordPress admin.
 * IMPORTANT: This function now relies exclusively on Advanced Custom Fields (ACF)
 * to retrieve custom data.
 */
function bacf_add_contact_columns( $columns ) {
    $new_columns = array(
        'contact_email'    => __( 'Email', 'bootstrap-advance-contact-form' ),
        'contact_phone'    => __( 'Phone', 'bootstrap-advance-contact-form' ),
        'contact_country'  => __( 'Country', 'bootstrap-advance-contact-form' ),
        'contact_company'  => __( 'Company', 'bootstrap-advance-contact-form' ),
        'contact_position' => __( 'Position', 'bootstrap-advance-contact-form' ),
        'contact_subject'  => __( 'Subject', 'bootstrap-advance-contact-form' ),
        'contact_message'  => __( 'Message', 'bootstrap-advance-contact-form' ), // Added for message content
    );

    // Insert new columns after the 'title' column.
    $offset = array_search( 'title', array_keys( $columns ) ) + 1;
    $columns = array_slice( $columns, 0, $offset, true ) + $new_columns + array_slice( $columns, $offset, null, true );

    return $columns;
}

/**
 * Populate custom columns in the 'Contact Messages' list table.
 * IMPORTANT: This function now relies exclusively on Advanced Custom Fields (ACF)
 * to retrieve custom data.
 */
function bacf_populate_contact_columns( $column, $post_id ) {
    // Ensure ACF is active before trying to get fields.
    if ( ! function_exists( 'get_field' ) ) {
        error_log( 'ACF is NOT active. Contact message custom columns cannot be populated for post ID: ' . $post_id . '. Please install and activate Advanced Custom Fields plugin.' );
        return; // Exit if ACF is not active.
    }

    // Use a switch statement to handle different custom columns.
    switch ( $column ) {
        case 'contact_email':
            echo esc_html( get_field( 'contact_email', $post_id ) );
            break;
        case 'contact_phone':
            $phone_value = get_field( 'contact_phone', $post_id );
            error_log('BACF AJAX: Retrieving phone for post ID ' . $post_id . ': ' . $phone_value); // Debug log for phone value
            echo esc_html( $phone_value );
            break;
        case 'contact_country':
            echo esc_html( get_field( 'contact_country', $post_id ) );
            break;
        case 'contact_company':
            echo esc_html( get_field( 'contact_company', $post_id ) );
            break;
        case 'contact_position':
            echo esc_html( get_field( 'contact_position', $post_id ) );
            break;
        case 'contact_subject':
            echo esc_html( get_field( 'contact_subject', $post_id ) );
            break;
        case 'contact_message': // Added case for message content
            // Display only a snippet of the message in the table for brevity
            $message_content = get_field( 'contact_message', $post_id );
            echo esc_html( wp_trim_words( $message_content, 15, '...' ) );
            break;
    }
}

/**
 * Add new submenu pages under the 'Contact Messages' CPT.
 */
function bacf_add_settings_page() {
    add_submenu_page(
        'edit.php?post_type=bs_contact_cpt', // Parent slug (our custom post type)
        __('Contact Form Settings', 'bootstrap-advance-contact-form'), // Page title
        __('Email Settings', 'bootstrap-advance-contact-form'), // Menu title
        'manage_options', // Capability required to access the page
        'bacf-settings', // Menu slug
        'bacf_settings_page_content' // Callback function to display the page content
    );

    // Add How To Use page
    add_submenu_page(
        'edit.php?post_type=bs_contact_cpt', // Parent slug (our custom post type)
        __('How To Use Contact Form', 'bootstrap-advance-contact-form'), // Page title
        __('How To Use', 'bootstrap-advance-contact-form'), // Menu title
        'manage_options', // Capability required to access the page
        'bacf-how-to-use', // Menu slug
        'bacf_how_to_use_page_content' // Callback function to display the page content
    );
}

/**
 * Register settings for the plugin.
 */
function bacf_register_settings() {
    // Register a new setting for our plugin
    register_setting(
        'bacf_settings_group', // Option group
        'bacf_admin_email_recipient', // Option name (this is where the email will be stored)
        'sanitize_email' // Sanitize callback function
    );

    // Add a settings section
    add_settings_section(
        'bacf_email_settings_section', // ID
        __('Admin Email Notification', 'bootstrap-advance-contact-form'), // Title
        'bacf_email_settings_section_callback', // Callback for section content
        'bacf-settings', // Page slug
    );

    // Add a settings field
    add_settings_field(
        'bacf_admin_email_field', // ID
        __('Recipient Email', 'bootstrap-advance-contact-form'), // Title
        'bacf_admin_email_field_callback', // Callback for field content
        'bacf-settings', // Page slug
        'bacf_email_settings_section', // Section ID
        array(
            'label_for' => 'bacf_admin_email_recipient',
            'class'     => 'bacf-admin-email-row',
            'name'      => 'bacf_admin_email_recipient',
        )
    );
}

/**
 * Callback for the settings section.
 */
function bacf_email_settings_section_callback() {
    echo '<p>' . __('Enter the email address where you want to receive contact form submissions.', 'bootstrap-advance-contact-form') . '</p>';
}

/**
 * Callback for the email settings field.
 */
function bacf_admin_email_field_callback($args) {
    // Get the current value of the option
    $email = get_option($args['name']);
    ?>
    <input type="email" id="<?php echo esc_attr($args['name']); ?>" name="<?php echo esc_attr($args['name']); ?>" value="<?php echo esc_attr($email); ?>" class="regular-text" placeholder="<?php esc_attr_e('e.g., your-email@example.com', 'bootstrap-advance-contact-form'); ?>" />
    <p class="description">
        <?php _e('This email address will receive all new contact form submissions. If left empty, the default WordPress admin email will be used.', 'bootstrap-advance-contact-form'); ?>
    </p>
    <?php
}

/**
 * Display the content of the plugin settings page.
 */
function bacf_settings_page_content() {
    ?>
    <div class="wrap">
        <h1><?php _e('Contact Form Settings', 'bootstrap-advance-contact-form'); ?></h1>
        <form action="options.php" method="post">
            <?php
            settings_fields('bacf_settings_group'); // Output security fields for the settings group
            do_settings_sections('bacf-settings'); // Output settings sections and fields
            submit_button(); // Output save button
            ?>
        </form>
    </div>
    <?php
}

/**
 * Display the content of the plugin's "How To Use" page.
 */
function bacf_how_to_use_page_content() {
    ?>
    <div class="wrap">
        <h1><?php _e('How To Use Bootstrap Advance Contact Form', 'bootstrap-advance-contact-form'); ?></h1>
        <p><?php _e('To display the contact form on any page or post, simply use the following shortcode:', 'bootstrap-advance-contact-form'); ?></p>
        <pre><code>[bootstrap-advance-contact-form-wp-shortcode]</code></pre>
        <p><?php _e('Paste this shortcode into the content editor of the desired page or post.', 'bootstrap-advance-contact-form'); ?></p>
        
        <h2><?php _e('Required Advanced Custom Fields (ACF) Setup', 'bootstrap-advance-contact-form'); ?></h2>
        <p><?php _e('This plugin relies on the Advanced Custom Fields (ACF) plugin to store contact form data. Please ensure ACF is installed and active.', 'bootstrap-advance-contact-form'); ?></p>
        <p><?php _e('You need to create a Field Group in ACF and assign it to the "Contact Messages" Custom Post Type. Below are the exact field names (keys) and recommended types you should use:', 'bootstrap-advance-contact-form'); ?></p>
        
        <table class="wp-list-table widefat fixed striped">
            <thead>
                <tr>
                    <th><?php _e('ACF Field Name (Key)', 'bootstrap-advance-contact-form'); ?></th>
                    <th><?php _e('Recommended Field Type', 'bootstrap-advance-contact-form'); ?></th>
                    <th><?php _e('Description', 'bootstrap-advance-contact-form'); ?></th>
                </tr>
            </thead>
            <tbody>
                <tr><td><code>contact_firstname</code></td><td>Text</td><td>Sender's first name.</td></tr>
                <tr><td><code>contact_lastname</code></td><td>Text</td><td>Sender's last name.</td></tr>
                <tr><td><code>contact_phone</code></td><td>Text</td><td>Sender's phone number.</td></tr>
                <tr><td><code>contact_email</code></td><td>Email</td><td>Sender's email address.</td></tr>
                <tr><td><code>contact_country</code></td><td>Select</td><td>Sender's country.</td></tr>
                <tr><td><code>contact_company</code></td><td>Text</td><td>Sender's company name.</td></tr>
                <tr><td><code>contact_position</code></td><td>Text</td><td>Sender's position.</td></tr>
                <tr><td><code>contact_subject</code></td><td>Text</td><td>Subject of the message.</td></tr>
                <tr><td><code>contact_message</code></td><td>Text Area</td><td>Full message content.</td></tr>
            </tbody>
        </table>
        
        <h2><?php _e('Email Notification Settings', 'bootstrap-advance-contact-form'); ?></h2>
        <p><?php _e('You can configure the recipient email address for contact form notifications under the "Email Settings" submenu item.', 'bootstrap-advance-contact-form'); ?></p>
    </div>
    <?php
}

/**
 * Add a dashboard widget to display the latest contact messages.
 */
function bacf_add_dashboard_widgets() {
    // Check if ACF is active before adding the widget.
    if ( ! function_exists( 'get_field' ) ) {
        return; // Do not add the widget if ACF is not active.
    }

    // Query to check if there are any contact messages.
    $check_messages_query = new WP_Query( array(
        'post_type'      => 'bs_contact_cpt',
        'posts_per_page' => 1, // Only need to check for one post to know if any exist.
        'post_status'    => 'publish',
        'fields'         => 'ids', // Only retrieve post IDs for efficiency.
    ) );

    if ( $check_messages_query->have_posts() ) {
        // If there are messages, add the dashboard widget.
        wp_add_dashboard_widget(
            'bacf_latest_contact_messages',         // Widget slug.
            __( 'Latest Contact Messages', 'bootstrap-advance-contact-form' ), // Title.
            'bacf_latest_contact_messages_callback' // Display function.
        );
    }
    // No need for an else block; if no posts, the widget simply isn't added.
}

/**
 * Callback function to display the content of the dashboard widget.
 */
function bacf_latest_contact_messages_callback() {
    // Check if ACF is active before trying to get fields.
    if ( ! function_exists( 'get_field' ) ) {
        echo '<p>' . __( 'Advanced Custom Fields plugin is not active. Please install and activate it to view contact messages here.', 'bootstrap-advance-contact-form' ) . '</p>';
        return;
    }

    // The number of items to show is now fixed at 3.
    $num_items = 3; 

    // Query the latest contact messages.
    $args = array(
        'post_type'      => 'bs_contact_cpt',
        'posts_per_page' => $num_items,
        'post_status'    => 'publish',
        'orderby'        => 'date',
        'order'          => 'DESC', // Latest to oldest
    );

    $contact_messages = new WP_Query( $args );

    if ( $contact_messages->have_posts() ) {
        echo '<ul class="bacf-dashboard-messages">';
        while ( $contact_messages->have_posts() ) {
            $contact_messages->the_post();
            $post_id = get_the_ID();
            $sender_firstname = get_field( 'contact_firstname', $post_id );
            $sender_lastname = get_field( 'contact_lastname', $post_id );
            $sender_email = get_field( 'contact_email', $post_id );
            $message_subject = get_field( 'contact_subject', $post_id ); // Get the subject field
            $edit_post_link = get_edit_post_link( $post_id );

            echo '<li>';
            echo '<strong>' . esc_html( $sender_firstname . ' ' . $sender_lastname ) . '</strong>';
            echo ' (' . esc_html( $sender_email ) . ')';
            echo '<br><strong>' . __( 'Subject:', 'bootstrap-advance-contact-form' ) . '</strong> ' . esc_html( $message_subject ); // Display the subject
            echo '<br><br><a href="' . esc_url( $edit_post_link ) . '" class="button button-primary button-small">' . __( 'Read Message', 'bootstrap-advance-contact-form' ) . '</a>';
            echo '<hr></li>';
        }
        echo '</ul>';
        wp_reset_postdata(); // Restore original Post Data
    } else {
        // This block will technically not be reached if the widget is only added when posts exist,
        // but it's good practice to have a fallback message if somehow it does.
        echo '<p>' . __( 'No contact messages found.', 'bootstrap-advance-contact-form' ) . '</p>';
    }
}