/**
 * Front-end JavaScript for Bootstrap Advance Contact Form plugin.
 * Handles AJAX form submission and intl-tel-input initialization,
 * including Bootstrap client-side validation.
 */

jQuery(document).ready(function($) {
	const phoneInputs = document.querySelectorAll(".internationalphone");
	let originalButtonText = '<button class="btn btn-primary" type="submit" name="submit" value="submit"><?php _e("Submit", "bootstrap-advance-contact-form"); ?></button>'; // Declare outside the submit handler to ensure its accessible

	// Function to initialize intl-tel-input for all phone input fields
	function initializeIntlTelInputInstances() {
		phoneInputs.forEach(function(input) {
			if(typeof window.intlTelInput !== 'undefined') {
				window.intlTelInput(input, {
					autoHideDialCode: false,
					initialCountry: 'auto',
					// utilsScript is specified here for clarity, but its loading is managed by PHP enqueue dependencies.
					// This path should match the one enqueued in your PHP.
					//utilsScript: "https://cdnjs.cloudflare.com/ajax/libs/intl-tel-input/17.0.8/js/utils.js"
				});
				console.log("BACF JS: intlTelInput initialized for:", input);
			} else {
				console.warn("BACF JS: intlTelInput is not defined. Phone input functionality may be limited. Check if intlTelInput.min.js and utils.js are enqueued correctly in PHP.");
			}
		});
	}

	// Delay the initialization of intl-tel-input slightly to ensure utils.js is fully loaded.
	// This is a common workaround for race conditions with asynchronously loaded scripts.
	setTimeout(function() {
		initializeIntlTelInputInstances();
		console.log("BACF JS: intlTelInput instances initialized after slight delay.");
	}, 1000); // 200ms delay, can be adjusted if needed


	// Use event delegation on the document for form submission.
	// This ensures that event listeners are attached to all forms with the specified class,
	// even if they are added dynamically to the DOM after the page loads.
	$(document).on('submit', '.bootstrap-advanced-contact-form__form', function(e) {
		e.preventDefault(); // Prevent the default browser form submission.

		const $form = $(this);
		const form = $form[0]; // Get the native DOM form element for `checkValidity()`.
		const $messagesDiv = $form.find('.bacf-form-messages'); // Message div within THIS form.
		const $submitButton = $form.find('button[type="submit"]');
		originalButtonText = $submitButton.html(); // Store original button text when form is submitted.

		// Clear any previous messages and remove validation classes.
		$messagesDiv.empty().removeClass('alert alert-success alert-warning alert-danger');
		$form.removeClass('was-validated'); // Remove Bootstrap's validation class

		// --- DEBUGGING: Log native HTML5 validation result ---
		const isFormNativeValid = form.checkValidity();
		console.log('BACF JS: Native HTML5 Form Validity:', isFormNativeValid);

		// --- Bootstrap Client-Side Validation ---
		// Check if the form is valid using native HTML5 validation API.
		if(!isFormNativeValid) {
			e.stopPropagation(); // Stop event propagation if form is invalid.
			$form.addClass('was-validated'); // Add Bootstrap's class to show validation feedback.
			// Display a generic validation error message.
			$messagesDiv.addClass('alert alert-danger').html(bacf_ajax_object.messages.validation_error);
			return; // Stop execution if form is invalid.
		}

		// Validate international phone number using intl-tel-input API.
		// Find the specific intlTelInput instance for this form's phone input.
		const $phoneInput = $form.find('.internationalphone');
		const intlTelInstance = window.intlTelInputGlobals ? window.intlTelInputGlobals.getInstance($phoneInput[0]) : null;

		// --- DEBUGGING: Log intl-tel-input instance and validity ---
		console.log('BACF JS: intlTelInput Instance:', intlTelInstance);
		if(intlTelInstance) {
			const isPhoneValid = intlTelInstance.isValidNumber();
			console.log('BACF JS: intlTelInput Phone Validity:', isPhoneValid);
			console.log('BACF JS: Formatted Phone Number:', intlTelInstance.getNumber());

			if(!isPhoneValid) {
				e.stopPropagation(); // Stop event propagation.
				$form.addClass('was-validated'); // Show validation feedback.
				// Add specific invalid feedback for the phone input.
				$phoneInput.addClass('is-invalid').siblings('.invalid-feedback').html(bacf_ajax_object.messages.validation_error);
				$messagesDiv.addClass('alert alert-danger').html(bacf_ajax_object.messages.validation_error);
				return; // Stop execution if phone number is invalid.
			} else {
				// If valid, ensure it's marked as valid.
				$phoneInput.removeClass('is-invalid').addClass('is-valid');
			}
		} else {
			console.warn("BACF JS: intlTelInput instance not found for phone input. Skipping intl-tel-input validation. This might happen if intlTelInput.min.js or its dependencies failed to load.");
			// If intlTelInput is not initialized, we might still want to proceed or show a different error.
			// For now, we'll let native validation handle it if intlTelInput isn't ready.
			// If this warning persists, it indicates intlTelInput.min.js or utils.js isn't loading correctly via PHP.
		}


		// Disable the submit button and show loading indicator.
		$submitButton.prop('disabled', true).html('<span class="spinner-border spinner-border-sm" role="status" aria-hidden="true"></span> ' + bacf_ajax_object.messages.submitting);

		// Serialize form data.
		const formData = $form.serializeArray(); // Use serializeArray to easily add custom fields.

		// Add the full international phone number from intl-tel-input to formData.
		// This ensures the server receives the correctly formatted number.
		if(intlTelInstance && intlTelInstance.isValidNumber()) {
			formData.push({name: 'myinternationalphone', value: intlTelInstance.getNumber()});
		}

		// Add nonce field to formData.
		formData.push({name: 'bacf_form_nonce_field', value: bacf_ajax_object.nonce});
		formData.push({name: 'action', value: 'bacf_submit_form'}); // Add AJAX action

		// --- DEBUGGING: Log formData before sending ---
		console.log('BACF JS: Sending formData:', formData);

		// Perform AJAX request.
		$.ajax({
			url: bacf_ajax_object.ajax_url, // WordPress AJAX endpoint.
			type: 'POST',
			data: formData, // Send serialized array.
			dataType: 'json', // Explicitly tell jQuery to expect JSON
			success: function(response) {
				// --- DEBUGGING: Log raw and parsed server response ---
				console.log('BACF JS: Raw Server Response (Success Callback):', response);

				if(response.success) {
					$messagesDiv.addClass('alert alert-success').html(response.data.message);
					form.reset(); // Reset native form fields.
					$form.removeClass('was-validated'); // Remove validation state.
					// Reset intl-tel-input instance after successful submission.
					// This is important to clear the visual state of the phone input
					// AND set the country to USA (+1).
					$form.find('.internationalphone').each(function() {
						const instance = window.intlTelInputGlobals.getInstance(this);
						if(instance) {
							instance.setNumber('+1'); // Clear the number.
							instance.setCountry('us'); // Set to USA (+1) after submission.
						}
					});
				} else {
					// This block is for when the server sends a JSON response with success: false
					const alertType = response.data.type || 'danger';
					$messagesDiv.addClass('alert alert-' + alertType).html(response.data.message);
					$form.addClass('was-validated'); // Keep validation feedback visible on error.
				}
			},
			error: function(jqXHR, textStatus, errorThrown) {
				// This callback function executes if the AJAX request itself fails
				// (e.g., network error, server not responding, HTTP status other than 2xx,
				// or if the response is not valid JSON when dataType is 'json').
				console.error('BACF JS: AJAX Error:', textStatus, errorThrown, jqXHR);
				// Log the raw response text for deeper inspection if JSON parsing failed
				console.error('BACF JS: Raw Response Text (Error Callback):', jqXHR.responseText);
				$messagesDiv.addClass('alert alert-danger').html(bacf_ajax_object.messages.error);
				$form.addClass('was-validated'); // Keep validation feedback visible on error.
			},
			complete: function() {
				// Re-enable button and restore original text.
				$submitButton.prop('disabled', false).html(originalButtonText);
			}
		});
	});
});
